// Copyright (C) 2015 by Thomas Carraro (1) and Sven Wetterauer (2)
// //
// // (1) Institute for Applied Mathematics, Heidelberg University
// //     E-mail: thomas.carraro@iwr.uni-heidelberg.de
// // (2) Institute for Applied Mathematics, Heidelberg University
// //     E-Mail: sven.wetterauer@iwr.uni-heidelberg.de
// //
// // This file is subject to the GNU Lesser General Public License
// // (LGPL) as published by the Free Software Foundation; either
// // version 2.1 of the License, or (at your option) any later
// // version. Please refer to the webpage http://www.dealii.org/
// // under the link License for the text and further information
// // on this license. You can use it, redistribute it, and/or
// // modify it under the terms of this licence.

#include <deal.II/base/quadrature_lib.h>
#include <deal.II/dofs/dof_handler.h>
#include <deal.II/grid/grid_tools.h>
#include <deal.II/lac/vector.h>
#include <deal.II/fe/fe_values.h>
#include <fstream>
#include <iostream>

#include"level_set.h"
#include"xfem_functions.h"

using namespace dealii;


// This sign function is used only in the visualization function. We use the value 1.e-16 instead of 0 to avoid a wrong behaviour due to round-off errors. The wrong behaviour can occur in case the level set function does intersect exactly a vertex, but it is calculated with a round-off error.
template<int dim>
double Xfem<dim>::sign(const double x)
{
  if (x< -1.e-16) return -1;
  else if (x> 1.e-16) return 1;
  else return 0;
}


template <int dim>
bool Xfem<dim>::interface_intersects_cell (const typename Triangulation<dim>::cell_iterator &cell) const
{
//A cell is cut by the interface if and only if there are vertices with different signs of the level set function. In order to determine if a cell is cut, we test if there is a change in sign between two vertices.
  for (unsigned int v=0; v<GeometryInfo<dim>::vertices_per_cell-1; ++v)
    if (ls.level_set(cell->vertex(v)) * ls.level_set(cell->vertex(v+1)) < 0)
      return true;
  if (ls.level_set(cell->vertex(0)) * ls.level_set(cell->vertex(GeometryInfo<dim>::vertices_per_cell-1))<0)
    return true;
// We only get here, if all vertices have the same sign, meaning the cell is not cut.
  return false;
}

// All cells surrounding an intersected cell are blending cells. The following function sets up a vector containing all direct neighbors and all diagonal neighbors of a cell.
template<int dim>
std::vector<typename hp::DoFHandler<dim>::active_cell_iterator> Xfem<dim>::find_all_neighbors(const typename hp::DoFHandler<dim>::active_cell_iterator &cell)
{
// Starting from the current cell, the diagonal neighbors can be reached by as neighbors of neighbors in 2 dimensions, and neighbors of neighbors of neighbors in 3 dimensions.
// In the first step, we define a set of all neighbors of neighbors (2d). Here a set is used in order to make sure no neighbor occurs twice.
  std::set<typename hp::DoFHandler<dim>::active_cell_iterator> adjacent_cells;

  for (unsigned int face=0; face<GeometryInfo<dim>::faces_per_cell; ++face)
    if (!cell->face(face)->at_boundary())
      {
        adjacent_cells.insert(cell->neighbor(face));
        if (dim>=2)
          for (unsigned int face2=0; face2<GeometryInfo<dim>::faces_per_cell; ++face2)
            if (!cell->neighbor(face)->face(face2)->at_boundary())
              {
                adjacent_cells.insert(cell->neighbor(face)->neighbor(face2));
                if (dim>=3)
                  for (unsigned int face3=0; face3<GeometryInfo<dim>::faces_per_cell; ++face3)
                    {
                      if (!cell->neighbor(face)->neighbor(face2)->face(face3)->at_boundary())
                        adjacent_cells.insert(cell->neighbor(face)->neighbor(face2)->neighbor(face3));
                    }
              }
      }

// The above set consists of all neighboring (direct and diagonal) and some additional cells.
// All neighboring cells have at least a common vertex with the current cell. We use this fact to save all neighboring cells in a vector.
  std::vector<typename hp::DoFHandler<dim>::active_cell_iterator> result;
  typename std::set<typename hp::DoFHandler<dim>::active_cell_iterator>::iterator cell_it=adjacent_cells.begin(),
                                                                                  endc_it=adjacent_cells.end();
  for (; cell_it!=endc_it; ++cell_it)
    {
      for (unsigned int v=0; v<GeometryInfo<dim>::vertices_per_cell; ++v)
        {
          unsigned int vertex_index=cell->vertex_index(v);
          for (unsigned int v2=0; v2<GeometryInfo<dim>::vertices_per_cell; ++v2)
            if ((*cell_it)->vertex_index(v2)==vertex_index)
              {
                result.push_back(*cell_it);
                goto next_cell;
              }
        }
next_cell:
      ;
    }

  return result;
}


// In order to define the ramp-function in a blending cell correctly, we need to know which DoFs have a shape function with support in intersected cells.
// This is done with the help of the vertex_has_intersected_neighbor function.
template<int dim>
bool Xfem<dim>::vertex_has_intersected_neighbor(const typename hp::DoFHandler<dim>::active_cell_iterator &cell,unsigned int vertex_index)
{
// The first part of this function is the same as in the find_all_neighbors function.
  std::set<typename hp::DoFHandler<dim>::active_cell_iterator> adjacent_cells;

  for (unsigned int face=0; face<GeometryInfo<dim>::faces_per_cell; ++face)
    if (!cell->face(face)->at_boundary())
      {
        adjacent_cells.insert(cell->neighbor(face));
        if (dim>=2)
          for (unsigned int face2=0; face2<GeometryInfo<dim>::faces_per_cell; ++face2)
            if (!cell->neighbor(face)->face(face2)->at_boundary())
              {
                adjacent_cells.insert(cell->neighbor(face)->neighbor(face2));
                if (dim>=3)
                  for (unsigned int face3=0; face3<GeometryInfo<dim>::faces_per_cell; ++face3)
                    {
                      if (!cell->neighbor(face)->neighbor(face2)->face(face3)->at_boundary())
                        adjacent_cells.insert(cell->neighbor(face)->neighbor(face2)->neighbor(face3));
                    }
              }
      }
// In the second part of this function we check, if there is an intersected cell with the same vertex_index as the vertex to which the DoF belongs.
  typename std::set<typename hp::DoFHandler<dim>::active_cell_iterator>::iterator cell_it=adjacent_cells.begin(),
                                                                                  endc_it=adjacent_cells.end();
  for (; cell_it!=endc_it; ++cell_it)
    {
      for (unsigned int v=0; v<GeometryInfo<dim>::vertices_per_cell; ++v)
        if ((*cell_it)->vertex_index(v)==vertex_index)
          {
            if (interface_intersects_cell(*cell_it))
              return true;
            goto next_cell;
          }
next_cell:
      ;
    }

  return false;
}


// In the blending cells (only for the weak discontinuity), we need the common DoFs that belong also to a cut cell. This function determines these DoFs by testing, for all adjacent cells of a DoF, if one of them is intersected by the interface.
template <int dim>
std::vector<unsigned int> Xfem<dim>::indices_of_ramp (const typename hp::DoFHandler<dim>::active_cell_iterator &cell)
{
// "indices" is a vector holding all standard DoFs which have an intersected adjacent cell.
  std::vector<unsigned int> indices;

  for (unsigned int i=0; i<cell->get_fe().dofs_per_cell; ++i)
    if (cell->get_fe().system_to_component_index(i).first==0)
      if (vertex_has_intersected_neighbor(cell,cell->vertex_index(cell->get_fe().system_to_component_index(i).second)))
        indices.push_back(i);

  return indices;
}


// The compute_quadrature function consists of two parts. First the subdivison of a cell is calculated. Second, a new quadrature rule for this subdivision is computed.
template <int dim>
Quadrature<dim> Xfem<dim>::compute_quadrature (const Quadrature<dim> &plain_quadrature,
                                               const typename hp::DoFHandler<dim>::active_cell_iterator &cell)
{
// subcells is a vector of cells, where every cell consists of a vector of four points, i.e. the vertices of the cell.
  std::vector<std::vector<Point<dim> > > subcells=calculate_subcells(cell);

  std::vector<Point<dim> > quad_points;
  std::vector<double>      quad_weights;

// This for-loop creates a quadrature formula that integrates discontinuous functions over the unit cell using its subcells.
  for (unsigned int i=0; i<subcells.size(); ++i)
//The function append_quadrature creates quadrature points and weights for a subcell of the unit cell. It is called iteratively so that the quadrature points and weights of the next subcell are added to the vectors quad_points and quad_weights.
    append_quadrature(plain_quadrature,subcells[i],quad_points,quad_weights);

  Quadrature<dim> new_quadrature(subcells.size()*plain_quadrature.size());
  new_quadrature.initialize(quad_points,quad_weights);

  return new_quadrature;
}

//This function creates a vector, containing all subcells, where every subcell is represented as a vector containing the four vertices of the subcell. The subdivison is generated on the unit cell.

template <int dim>
std::vector< std::vector<Point<dim> > > Xfem<dim>::calculate_subcells
(const typename hp::DoFHandler<dim>::active_cell_iterator &cell)
{
  MappingQ1<dim> mapping;
  Level_Set<dim> ls;
  std::vector< std::vector<Point<dim> > > subcells;

  std::vector<double> level_set_values (GeometryInfo<dim>::vertices_per_cell);
  for (unsigned int v=0; v<GeometryInfo<dim>::vertices_per_cell; ++v)
    level_set_values[v] = ls.level_set (cell->vertex(v));

  unsigned int type = 0;
//A cell can be intersected by the interface in different ways. For each intersection type a different subdivison of the cell is needed. The subcells used in this program are quadrilaterals in two dimensions (hexahedra will be used in the three dimensional case). Including the case of uncut cells, there are 5 different types in 2d and 15 in 3d (3d case not implemented here). The type of the cut is uniquely determined by the absolute values of the sum and the product of the sign of the level set function in the vertices:
  // type 1: not intersected
  // type 2: -+++, +-++, ++-+, +++-, +---, -+--, --+-, ---+
  // type 3: +--+, ++--, +-+-, -++-, --++, -+-+
  // type 4: 0+--, 0-++, 0+-+, 0-+-, -0++, +0--, +0+-, -0-+,
  //         +-0-, -+0-, ++0-, --0+, +-+0, -+-0, ++-0, --+0
  // type 5: 0+0-, 0-0+, +0-0, -0+0
  int sign_ls[GeometryInfo<dim>::vertices_per_cell];
  int sum =0;
  int prod=1;
  for (unsigned int v=0; v<GeometryInfo<dim>::vertices_per_cell; ++v)
    {
      if (level_set_values[v] > 1.e-16) sign_ls[v] = 1;
      else if (level_set_values[v] < -1.e-16) sign_ls[v] = -1;
      else sign_ls[v] = 0;

      sum+= sign_ls[v];
      prod *= sign_ls[v];
    }

  // If the product is !=0 the interface does not cut a node. Using the sum of level set values the different types can be determined.
  int sum_abs=std::fabs(sum);
  switch (dim)
    {
    case 2:
      if (prod !=0)
        {
          switch (sum_abs)
            {
            case 4:
              type=1;
              break;
            case 2:
              type=2;
              break;
            case 0:
              type=3;
              break;
            default:
              assert(0);
            }
        }
      else
        {
          switch (sum_abs)
            {
            case 3:
              type=1;
              break;
            case 2:
              type=1;
              break;
            case 1:
              type=4;
              break;
            case 0:
              type=5;
              break;
            default:
              assert(0);
            }
        }
      break;

    default:
      std::cout<<"This program only works in 2d!!!!"<<std::endl;
      assert(0);
    }

  Point<dim> v0 (0,0);
  Point<dim> v1 (1,0);
  Point<dim> v2 (0,1);
  Point<dim> v3 (1,1);

  Point<dim> A (0,0);
  Point<dim> B (0,0);
  Point<dim> C (0,0);
  Point<dim> D (0,0);
  Point<dim> E (0,0);
  Point<dim> F (0,0);
  Point<dim> G (0,0);
  Point<dim> H (0,0);
  Point<dim> I (0,0);

//For type==1 the cell is not intersected by the interface, meaning that we do not need a special quadrature rule. Therefore, no subdivison is needed and the only subcell returned is the unit cell itself.
  if (type == 1)
    {
      std::vector<Point<dim> > subcell_vertices(4);
      subcell_vertices[0] = v0;
      subcell_vertices[1] = v1;
      subcell_vertices[2] = v2;
      subcell_vertices[3] = v3;

      subcells.push_back(subcell_vertices);
    }

//The second type describes the case that the interface intersects two neighboring faces.

  if (type ==2)
    {
//All types are described by a similar code. First, all points that are needed as vertices of a subcell are calculated for every different case that can occur.
      unsigned int Pos=0;
      if (sign_ls[0]!=sign_ls[1] && sign_ls[0]!=sign_ls[2] && sign_ls[0]!=sign_ls[3]) Pos = 0;
      else if (sign_ls[1]!=sign_ls[0] && sign_ls[1]!=sign_ls[2] && sign_ls[1]!=sign_ls[3]) Pos = 1;
      else if (sign_ls[2]!=sign_ls[0] && sign_ls[2]!=sign_ls[1] && sign_ls[2]!=sign_ls[3]) Pos = 2;
      else if (sign_ls[3]!=sign_ls[0] && sign_ls[3]!=sign_ls[1] && sign_ls[3]!=sign_ls[2]) Pos = 3;
      else assert(0);

      if (Pos == 0)
        {
//  2-------------3
//  |             |
//  B             |
//  | \           |
//  |  \          |
//  |   C         |
//  |   /\        |
//  F--D  \       |
//  |  |   \      |
//  0--E---A------1
          A = v1 + level_set_values[1]/(level_set_values[1]-level_set_values[0]) * (v0-v1);
          B = v2 + level_set_values[2]/(level_set_values[2]-level_set_values[0]) * (v0-v2);;
          C = 0.5 * (A + B);
          D = 1./3. * (A + B + v0);
          E = 0.5 * (v0 + A);
          F = 0.5 * (v0 + B);
        }
      else if (Pos == 1)
        {
//  2-------------3
//  |             |
//  |             B
//  |           / |
//  |          /  |
//  |         C   |
//  |        /\   |
//  |       /  D--F
//  |      /   |  |
//  0------A---E--1
          A = v0 + level_set_values[0]/(level_set_values[0]-level_set_values[1]) * (v1 - v0);
          B = v3 + level_set_values[3]/(level_set_values[3]-level_set_values[1]) * (v1 - v3);
          C = 0.5 * (A + B);
          D = 1./3. * (v1 + A + B);
          E = 0.5 * (v1 + A);
          F = 0.5 * (v1 + B);
        }

      else if (Pos == 2)
        {
//  2--E---A------3
//  |  |   /      |
//  F--D  /       |
//  |   \/        |
//  |   C         |
//  |  /          |
//  | /           |
//  B             |
//  |             |
//  0-------------1
          A = v3 + level_set_values[3]/(level_set_values[3]-level_set_values[2]) * (v2-v3);
          B = v0 + level_set_values[0]/(level_set_values[0]-level_set_values[2]) * (v2-v0);
          C = 0.5 * (A + B);
          D = 1./3. * (v2 + A + B);
          E = 0.5 * (v2 + A);
          F = 0.5 * (v2 + B);
        }
      else if (Pos == 3)
        {
//  2------A---E--3
//  |      \   |  |
//  |       \  D--F
//  |        \/   |
//  |         C   |
//  |          \  |
//  |           \ |
//  |             B
//  |             |
//  0-------------1
          A = v2 + level_set_values[2]/(level_set_values[2]-level_set_values[3]) * (v3-v2);
          B = v1 + level_set_values[1]/(level_set_values[1]-level_set_values[3]) * (v3-v1);
          C = 0.5 * (A + B);
          D = 1./3. * (v3 + A + B);
          E = 0.5 * (v3 + A);
          F = 0.5 * (v3 + B);
        }

//Next, these points including the vertices of the unit cell are enumerated.
      Point<dim> all_vertices[10];
      all_vertices[0] = v0;
      all_vertices[1] = v1;
      all_vertices[2] = v2;
      all_vertices[3] = v3;
      all_vertices[4] = A;
      all_vertices[5] = B;
      all_vertices[6] = C;
      all_vertices[7] = D;
      all_vertices[8] = E;
      all_vertices[9] = F;

// The lookup table for the subdivision contains all cases and the number of the vertices of every subcell.
      unsigned int subcell_v_indices[4][5][4] =
      {
        {{0,8,9,7}, {9,7,5,6}, {8,4,7,6}, {5,6,2,3}, {6,4,3,1}},
        {{8,1,7,9}, {4,8,6,7}, {6,7,5,9}, {0,4,2,6}, {2,6,3,5}},
        {{9,7,2,8}, {5,6,9,7}, {6,4,7,8}, {0,1,5,6}, {6,1,4,3}},
        {{7,9,8,3}, {4,6,8,7}, {6,5,7,9}, {0,6,2,4}, {0,1,6,5}}
      };

//With the help of the lookup table the subcells are determined and described as vectors of points into the return-vector.
      for (unsigned int subcell = 0; subcell<5; subcell++)
        {
          std::vector<Point<dim> > subcell_vertices;
          for (unsigned int i=0; i<4; i++)
            subcell_vertices.push_back( all_vertices[subcell_v_indices[Pos][subcell][i]] );
          subcells.push_back(subcell_vertices);
        }
    }

//In the third type, the interface intersects two opposing faces, meaning that two quadrilaterals are created.

  if (type == 3)
    {
      unsigned int Pos=0;
      if ( sign_ls[0]==sign_ls[1] && sign_ls[2]==sign_ls[3] )
        {
//  2-------------3
//  |             |
//  |             |
//  |             |
//  A-------------B
//  |             |
//  |             |
//  |             |
//  |             |
//  0-------------1
          Pos = 0;
          A = v0 + level_set_values[0]/((level_set_values[0]-level_set_values[2])) * (v2-v0);
          B = v1 + level_set_values[1]/((level_set_values[1]-level_set_values[3])) * (v3-v1);
        }
      else if ( sign_ls[0]==sign_ls[2] && sign_ls[1]==sign_ls[3] )
        {
//  2------A------3
//  |      |      |
//  |      |      |
//  |      |      |
//  |      |      |
//  |      |      |
//  |      |      |
//  |      |      |
//  |      |      |
//  0------B------1
          Pos = 1;
          A = v0 + level_set_values[0]/((level_set_values[0]-level_set_values[1])) * (v1-v0);
          B = v2 + level_set_values[2]/((level_set_values[2]-level_set_values[3])) * (v3-v2);
        }
      else if ( sign_ls[0]==sign_ls[3] && sign_ls[1]==sign_ls[2] )
        {
          std::cout << "Error: the element has two interfaces and this is not allowed" << std::endl;
          assert(0);
        }
      else
        {
          std::cout << "Error: the level set function has not the right values" << std::endl;
          assert(0);
        }

      Point<dim> all_vertices[6];
      all_vertices[0] = v0;
      all_vertices[1] = v1;
      all_vertices[2] = v2;
      all_vertices[3] = v3;
      all_vertices[4] = A;
      all_vertices[5] = B;

      unsigned int subcell_v_indices[2][2][4] =
      {
        {{0,1,4,5}, {4,5,2,3}},
        {{0,4,2,5}, {4,1,5,3}}
      };
      for (unsigned int subcell = 0; subcell<2; subcell++)
        {
          std::vector<Point<dim> > subcell_vertices;
          for (unsigned int i=0; i<4; i++)
            subcell_vertices.push_back( all_vertices[subcell_v_indices[Pos][subcell][i]] );
          subcells.push_back(subcell_vertices);
        }
    }

//In the forth and fifth type, the interface cuts through a vertex. In type==4 it passes exactly through one vertex.

  if (type == 4)
    {
      unsigned int Pos=0;

      if (sign_ls[0] == 0 && sign_ls[1]!=sign_ls[3]) Pos = 0;
      else if (sign_ls[0] == 0 && sign_ls[1]==sign_ls[3]) Pos = 1;
      else if (sign_ls[1] == 0 && sign_ls[0]!=sign_ls[2]) Pos = 2;
      else if (sign_ls[1] == 0 && sign_ls[0]==sign_ls[2]) Pos = 3;
      else if (sign_ls[2] == 0 && sign_ls[1]!=sign_ls[3]) Pos = 4;
      else if (sign_ls[2] == 0 && sign_ls[1]==sign_ls[3]) Pos = 5;
      else if (sign_ls[3] == 0 && sign_ls[0]!=sign_ls[2]) Pos = 6;
      else if (sign_ls[3] == 0 && sign_ls[0]==sign_ls[2]) Pos = 7;
      else assert(0);

      if (Pos == 0)
        {
//  2-------------3
//  |             |
//  |             B
//  |          /  |
//  |        /    |
//  |      E      |
//  |    /   \    |
//  |  /      F---D
//  |/        |   |
//  0---------C---1
          A=v0;
          B = v3 + level_set_values[3]/(level_set_values[3]-level_set_values[1]) * (v1-v3);
          C = 0.5 * (v0 + v1);
          D = 0.5 * (B + v1);
          E = 0.5 * (B + v0);
          F = 1./3. * (v0 + v1 + B);
        }
      else if (Pos == 1)
        {
//  2--D-----B----3
//  |  |    /     |
//  C--F   /      |
//  |   \ /       |
//  |    E        |
//  |   /         |
//  |  /          |
//  | /           |
//  |/            |
//  0-------------1
          A=v0;
          B = v3 + level_set_values[3]/(level_set_values[3]-level_set_values[2]) * (v2-v3);
          C = 0.5 * (v0 + v2);
          D = 0.5 * (B + v2);
          E = 0.5 * (B + v0);
          F = 1./3. * (v0 + v2 + B);
        }
      else if (Pos == 2)
        {
//  2-------------3
//  |             |
//  B             |
//  |  \          |
//  |    \        |
//  |      E      |
//  |    /   \    |
//  D---F      \  |
//  |   |        \|
//  0---C---------1
          A=v1;
          B = v2 + level_set_values[2]/(level_set_values[2]-level_set_values[0]) * (v0-v2);
          C = 0.5 * (v0 + v1);
          D = 0.5 * (B + v0);
          E = 0.5 * (B + v1);
          F = 1./3. * (v0 + v1 + B);
        }
      else if (Pos == 3)
        {
//  2----B-----D--3
//  |     \    |  |
//  |      \   F--C
//  |       \ /   |
//  |        E    |
//  |         \   |
//  |          \  |
//  |           \ |
//  |            \|
//  0-------------1
          A=v1;
          B = v3 + level_set_values[3]/(level_set_values[3]-level_set_values[2]) * (v2-v3);
          C = 0.5 * (v1 + v3);
          D = 0.5 * (B + v3);
          E = 0.5 * (B + v1);
          F = 1./3. * (v1 + v3 + B);
        }
      else if (Pos == 4)
        {
//  2---------C---3
//  |\        |   |
//  |  \      F---D
//  |    \   /    |
//  |      E      |
//  |        \    |
//  |          \  |
//  |             B
//  |             |
//  0-------------1
          A=v2;
          B = v3 + level_set_values[3]/(level_set_values[3]-level_set_values[1]) * (v1-v3);
          C = 0.5 * (v2 + v3);
          D = 0.5 * (B + v3);
          E = 0.5 * (B + v2);
          F = 1./3. * (v2 + v3 + B);
        }
      else if (Pos == 5)
        {
//  2-------------3
//  |\            |
//  | \           |
//  |  \          |
//  |   \         |
//  |    E        |
//  |   / \       |
//  C--F   \      |
//  |  |    \     |
//  0--D-----B----1
          A=v2;
          B = v1 + level_set_values[1]/(level_set_values[1]-level_set_values[0]) * (v0-v1);
          C = 0.5 * (v0 + v2);
          D = 0.5 * (B + v0);
          E = 0.5 * (B + v2);
          F = 1./3. * (v0 + v2 + B);
        }
      else if (Pos == 6)
        {
//  2---C---------3
//  |   |        /|
//  D---F      /  C
//  |    \   /    |
//  |      E      |
//  |    /        |
//  |  /          |
//  B             |
//  |             |
//  0-------------1
          A=v3;
          B = v2 + level_set_values[2]/(level_set_values[2]-level_set_values[0]) * (v0-v2);
          C = 0.5 * (v2 + v3);
          D = 0.5 * (B + v2);
          E = 0.5 * (B + v3);
          F = 1./3. * (v2 + v3 + B);
        }
      else if (Pos == 7)
        {
//  2-------------3
//  |            /|
//  |           / |
//  |          /  |
//  |         /   |
//  |        E    |
//  |       / \   |
//  |      /   F--C
//  |     /    |  |
//  0----B-----D--1
          A=v3;
          B = v1 + level_set_values[1]/(level_set_values[1]-level_set_values[0]) * (v0-v1);
          C = 0.5 * (v1 + v3);
          D = 0.5 * (B + v1);
          E = 0.5 * (B + v3);
          F = 1./3. * (v1 + v3 + B);
        }

      Point<dim> all_vertices[9];
      all_vertices[0] = v0;
      all_vertices[1] = v1;
      all_vertices[2] = v2;
      all_vertices[3] = v3;
      all_vertices[4] = B;
      all_vertices[5] = C;
      all_vertices[6] = D;
      all_vertices[7] = E;
      all_vertices[8] = F;

      // Lookup table for the decomposition
      unsigned int subcell_v_indices[8][4][4] =
      {
        {{5,1,8,6}, {8,6,7,4}, {0,5,7,8}, {0,4,2,3}},
        {{5,8,2,6}, {8,7,6,4}, {0,7,5,8}, {0,1,4,3}},
        {{0,5,6,8}, {6,8,4,7}, {5,1,8,7}, {4,1,2,3}},
        {{8,5,6,3}, {7,8,4,6}, {7,1,8,5}, {0,1,2,4}},
        {{8,6,5,3}, {7,4,8,6}, {7,8,2,5}, {0,1,2,4}},
        {{0,6,5,8}, {6,4,8,7}, {5,8,2,7}, {4,1,2,3}},
        {{6,8,2,5}, {4,7,6,8}, {8,7,5,3}, {0,1,4,3}},
        {{6,1,8,5}, {4,6,7,8}, {8,5,7,3}, {0,4,2,3}}
      };

      for (unsigned int subcell = 0; subcell<4; subcell++)
        {
          std::vector<Point<dim> > subcell_vertices;
          for (unsigned int i=0; i<GeometryInfo<dim>::vertices_per_cell; i++)
            subcell_vertices.push_back( all_vertices[subcell_v_indices[Pos][subcell][i]] );
          subcells.push_back(subcell_vertices);
        }
    }

//In the last case the interface subdivides the cell into two triangles, by intersecting two opposite vertices.

  if (type == 5)
    {
      unsigned int Pos=0;

      if ( sign_ls[0]==sign_ls[3] )
        {
//  2---E---------3
//  |   |        /|
//  F---H      /  |
//  |     \  /    |
//  |      G      |
//  |    /  \     |
//  |  /      I---D
//  |/        |   |
//  0---------C---1
          Pos = 0;
          A=v0;
          B=v3;
          C = 0.5 * (v0 + v1);
          D = 0.5 * (v1 + v3);
          E = 0.5 * (v2 + v3);
          F = 0.5 * (v0 + v2);
          G = 0.5 * (v0 + v3);
          H = 1./3. * (v0 + v2 + v3);
          I = 1./3. * (v0 + v1 + v3);
        }
      else if ( sign_ls[1]==sign_ls[2] )
        {
//  2---------E---3
//  |\        |   |
//  |  \      I---D
//  |    \  /     |
//  |      G      |
//  |     /  \    |
//  F---H      \  |
//  |   |        \|
//  0---C---------1
          Pos = 1;
          A=v1;
          B=v2;
          C = 0.5 * (v0 + v1);
          D = 0.5 * (v1 + v3);
          E = 0.5 * (v2 + v3);
          F = 0.5 * (v0 + v2);
          G = 0.5 * (v1 + v2);
          H = 1./3. * (v0 + v1 + v2);
          I = 1./3. * (v1 + v2 + v3);
        }
      else
        {
          std::cout << "Error: the level set function has not the right values" << std::endl;
          assert(0);
        }

      Point<dim> all_vertices[11];
      all_vertices[0] = v0;
      all_vertices[1] = v1;
      all_vertices[2] = v2;
      all_vertices[3] = v3;
      all_vertices[4] = C;
      all_vertices[5] = D;
      all_vertices[6] = E;
      all_vertices[7] = F;
      all_vertices[8] = G;
      all_vertices[9] = H;
      all_vertices[10] = I;

      unsigned int subcell_v_indices[2][6][4] =
      {
        {{0,4,8,10}, {4,1,10,5}, {10,5,8,3}, {0,8,7,9}, {9,8,6,3}, {7,9,2,6}},
        {{0,4,7,9}, {4,1,9,8}, {7,9,2,8}, {8,1,10,5}, {8,10,2,6}, {10,5,6,3}}
      };

      for (unsigned int subcell = 0; subcell<6; subcell++)
        {
          std::vector<Point<dim> > subcell_vertices;
          for (unsigned int i=0; i<4; i++)
            subcell_vertices.push_back( all_vertices[subcell_v_indices[Pos][subcell][i]] );
          subcells.push_back(subcell_vertices);
        }
    }

// Definition of the linearized level set function, which is needed for the visualization of the solution. Even if the exact level set function is used in the XFEM formulation, its linearization is used for visualization purposes.
// The linearized level set function has the formula: n0*x+n1*y-c

  Point<dim> real_A=mapping.transform_unit_to_real_cell(cell,A);
  Point<dim> real_B=mapping.transform_unit_to_real_cell(cell,B);
  n0=real_B[1]-real_A[1];
  n1=real_A[0]-real_B[0];
  c=n0*real_A[0]+n1*real_A[1];
  for (unsigned int v=0; v<GeometryInfo<dim>::vertices_per_cell; ++v)
    if (sign_ls[v]<0)
      if (n0*cell->vertex(v)[0]+n1*cell->vertex(v)[1]-c>0)
        {
          n0=-n0;
          n1=-n1;
          c=-c;
        }
//For the calculation of a line integral along the interface one needs the intersection points between the boundary of the cell and the interface.
  intersection_points.resize(2);
  intersection_points[0]=A;
  intersection_points[1]=B;
  return subcells;
}

template <int dim>
void Xfem<dim>::append_quadrature ( const Quadrature<dim> &plain_quadrature,
                                    const std::vector<Point<dim> > &v,
                                    std::vector<Point<dim> > &xfem_points,
                                    std::vector<double>      &xfem_weights)

{
// Project integration points into subelements.
// This maps quadrature points from a reference element to a subelement of a reference element.
// To implement the action of this map the coordinates of the subelements have been calculated (A(0)...F(0),A(1)...F(1))
// the coordinates of the quadrature points are given by the bi-linear map defined by the shape functions, i.e.
// $x^\prime_i = \sum_j v^\prime \phi_j(x^hat_i)$, where the $\phi_j$ are the shape functions of the FEQ.

  unsigned int n_v = GeometryInfo<dim>::vertices_per_cell;

  std::vector<Point<dim> > q_points = plain_quadrature.get_points();
  std::vector<double> W = plain_quadrature.get_weights();
  std::vector<double> phi(n_v);
  std::vector<Tensor<1,dim> > grad_phi(n_v);

  const unsigned int   n_q_points    = plain_quadrature.size();

  for ( unsigned int i = 0; i < n_q_points; i++)
    {
      switch (dim)
        {
        case 2:
        {
          double xi  = q_points[i](0);
          double eta = q_points[i](1);

// Define shape functions on reference element
// we consider a bi-linear mapping
          phi[0] = (1. - xi) * (1. - eta);
          phi[1] = xi * (1. - eta);
          phi[2] = (1. - xi) * eta;
          phi[3] = xi * eta;

          grad_phi[0][0] = (-1. + eta);
          grad_phi[1][0] = (1. - eta);
          grad_phi[2][0] = -eta;
          grad_phi[3][0] = eta;

          grad_phi[0][1] = (-1. + xi);
          grad_phi[1][1] = -xi;
          grad_phi[2][1] = 1-xi;
          grad_phi[3][1] = xi;

          break;
        }
        default:
          std::cout<<"Only the 2d case is implemented in this program"<<std::endl;
          assert(0);
        }

      Tensor<2,dim> jacobian;

// Calculate Jacobian of the transformation
      for (unsigned int d=0; d<dim; ++d)
        for (unsigned int e=0; e<dim; ++e)
          for (unsigned int j = 0; j<GeometryInfo<dim>::vertices_per_cell; j++)
            jacobian[d][e] += grad_phi[j][e] * v[j](d);

      double detJ = determinant(jacobian);
      xfem_weights.push_back (W[i] * detJ);

// Map integration points from reference element to subcell of reference element
      Point<dim> q_prime;
      for (unsigned int d=0; d<dim; ++d)
        for (unsigned int j = 0; j<GeometryInfo<dim>::vertices_per_cell; j++)
          q_prime[d] += v[j](d) * phi[j];
      xfem_points.push_back(q_prime);
    }
}


// The visualization of XFEM functions uses the subdivision in subcells of cut cells. Therefore, a mesh containing all uncut cells and all subcells of cut cells is created. To every vertex of this new mesh the value of the solution is assigned. Then the mesh and the solution values are passed to the write_plot function to write the vtk-file with this data.
template<int dim>
void Xfem<dim>::plot_vtk_strong ( hp::DoFHandler<dim> &dof_handler,
                                  hp::FECollection<dim> &fe_collection,
                                  Vector<double> &solution,
                                  const unsigned int cycle)
{
  Level_Set<dim> ls;
  std::vector<Point<dim>> mesh_;
  std::vector<double> solution_;

  typename hp::DoFHandler<dim>::active_cell_iterator cell=dof_handler.begin_active(),
                                                     endc=dof_handler.end();

  for (; cell!=endc; ++cell)
    {
      unsigned int dofs_per_cell=cell->get_fe().dofs_per_cell;
// For uncut cells, we save the vertices of the cell in the "mesh_" vector and the values of the solution in the vertices in the "solution_" vector.
      if (cell->active_fe_index()==0)
        {
          Vector<double> dof_values(dofs_per_cell);
// This function calculates the value of the solution at the DoFs of a cell, in our case at the vertices of the cell.
          cell->get_dof_values(solution,dof_values);

          for (unsigned int i=0; i<dofs_per_cell; ++i)
            if (cell->get_fe().system_to_component_index(i).first==0)
              {
                mesh_.push_back(cell->vertex(cell->get_fe().system_to_component_index(i).second));
                solution_.push_back(dof_values[i]);
              }
        }
      else
        {
// In case of intersected cells, we need the subdivision of this cell
          std::vector<std::vector<Point<dim> > > subcells=calculate_subcells(cell);

          Vector<double> dof_values(dofs_per_cell);
          cell->get_dof_values(solution,dof_values);

          for (unsigned int subcell=0; subcell<subcells.size(); ++subcell)
            {
// Then, we need the values of the shape functions at the vertices of the subcells. The FEValues object calculates these values in the given quadrature points. So we define a quadrature rule, in which the quadrature points coincide with the vertices of the subcells. The weights are not needed. Therefore, they can be chosen arbitrarily.
              Quadrature<dim> help_quad_formula;
              std::vector<Point<dim> > quad_points;
              std::vector<double> quad_weights;

              Assert(subcells[subcell].size()==GeometryInfo<dim>::vertices_per_cell,ExcInternalError());

              for (unsigned int v=0; v<GeometryInfo<dim>::vertices_per_cell; ++v)
                {
                  quad_points.push_back(subcells[subcell][v]);
                  quad_weights.push_back(1.);
                }

              help_quad_formula.initialize(quad_points,quad_weights);

              FEValues<dim> fe_values(fe_collection[1],help_quad_formula,
                                      update_values  |  update_quadrature_points);

              fe_values.reinit(cell);
// z is a testing parameter to determine on which side of the interface the current subcell is located.
// Due to round-off error the value of the linearized level set function in the vertices on the interface could have the wrong sign. Nevertheless, the sum of all values of the vertices maintains the correct sign.
              double z=0;

              for (unsigned int v=0; v<GeometryInfo<dim>::vertices_per_cell; ++v)
                z += level_set(fe_values.quadrature_point(v));
// In this loop the correct values of the solution are calculated at the vertices of the current subcell, i.e. at the quadrature points of the help_quad_formula. We cannot use the function get_function_values of the FEValues class, because this does not incorporate the additional shape functions. Therefore, we need to calculate the value of the solution explicitly in order to distinguish between standard and additional shape functions.
              for (unsigned int q_point=0; q_point<help_quad_formula.size(); ++q_point)
                {
                  double sol_in_q_point=0;

                  for (unsigned int i=0; i<dofs_per_cell; ++i)
                    if (cell->get_fe().system_to_component_index(i).first==0)
                      {
                        sol_in_q_point+= dof_values[i]
                                         * fe_values.shape_value(i,q_point);
                      }
                    else
                      {
                        if (z<0)
                          sol_in_q_point += dof_values[i]
                                            * fe_values.shape_value(i,q_point)
                                            *( -1-sign(ls.level_set(cell->vertex(cell->get_fe().system_to_component_index(i).second))));
                        else
                          sol_in_q_point += dof_values[i]
                                            * fe_values.shape_value(i,q_point)
                                            *( 1-sign(ls.level_set(cell->vertex(cell->get_fe().system_to_component_index(i).second))));
                      }

                  mesh_.push_back(fe_values.quadrature_point(q_point));
                  solution_.push_back(sol_in_q_point);
                }
            }
        }
    }
//write_plot writes the calculated data in a vtk-file, with name "solution-0'cycle'.vtk".
  std::string filename = "solution-";
  filename += ('0' + cycle);
  write_plot(mesh_,solution_,filename);
}


// The visualization of a weak discontinuity is basically the same as for the strong case. The only difference is in the additional basis functions.
template<int dim>
void Xfem<dim>::plot_vtk_weak ( hp::DoFHandler<dim> &dof_handler,
                                hp::FECollection<dim> &fe_collection,
                                Vector<double> &solution,
                                const unsigned int cycle)
{
  Level_Set<dim> ls;
  std::vector<Point<dim>> mesh_;
  std::vector<double> solution_;

  typename hp::DoFHandler<dim>::active_cell_iterator cell=dof_handler.begin_active(),
                                                     endc=dof_handler.end();

  for (; cell!=endc; ++cell)
    {
      unsigned int dofs_per_cell=cell->get_fe().dofs_per_cell;

      if (cell->active_fe_index()==0)
        {
          Vector<double> dof_values(dofs_per_cell);
          cell->get_dof_values(solution,dof_values);

          for (unsigned int i=0; i<dofs_per_cell; ++i)
            if (cell->get_fe().system_to_component_index(i).first==0)
              {
                mesh_.push_back(cell->vertex(cell->get_fe().system_to_component_index(i).second));
                solution_.push_back(dof_values[i]);
              }
        }
      else
        {
// There is no need to distinguish between intersected cells and blending cells. For blending cells the only subcell is the whole cell itself, meaning that we only need the values of the solution at the DoFs. At the vertices of the cell the additional shape functions are zero due to the shift. So only the standard part will have an influence on the visualization of the solution.
          std::vector<std::vector<Point<dim> > > subcells=calculate_subcells(cell);

          Vector<double> dof_values(dofs_per_cell);
          cell->get_dof_values(solution,dof_values);

          for (unsigned int subcell=0; subcell<subcells.size(); ++subcell)
            {
              Quadrature<dim> help_quad_formula;
              std::vector<Point<dim> > quad_points;
              std::vector<double> quad_weights;

              Assert(subcells[subcell].size()==GeometryInfo<dim>::vertices_per_cell,ExcInternalError());

              for (unsigned int v=0; v<GeometryInfo<dim>::vertices_per_cell; ++v)
                {
                  quad_points.push_back(subcells[subcell][v]);
                  quad_weights.push_back(1.);
                }

              help_quad_formula.initialize(quad_points,quad_weights);

              FEValues<dim> fe_values(fe_collection[1],help_quad_formula,
                                      update_values  |  update_quadrature_points);

              fe_values.reinit(cell);

              for (unsigned int q_point=0; q_point<help_quad_formula.size(); ++q_point)
                {
                  double sol_in_q_point=0;

                  for (unsigned int i=0; i<dofs_per_cell; ++i)
                    if (cell->get_fe().system_to_component_index(i).first==0)
                      {
                        sol_in_q_point+= dof_values[i]
                                         * fe_values.shape_value(i,q_point);
                      }
                    else
                      {
                        sol_in_q_point += dof_values[i]
                                          * fe_values.shape_value(i,q_point)
                                          *( std::fabs(ls.level_set(fe_values.quadrature_point(q_point)))
                                             - std::fabs(ls.level_set(cell->vertex(cell->get_fe().system_to_component_index(i).second))));
                      }
                  mesh_.push_back(fe_values.quadrature_point(q_point));
                  solution_.push_back(sol_in_q_point);
                }
            }
        }
    }
  std::string filename = "solution-";
  filename += ('0' + cycle);
  write_plot(mesh_,solution_,filename);
}

// The purpose of this function is to write a vtk-file which represents the given mesh with the solution passed as argument. The structure of a vtk-file is explained in http://www.vtk.org.
template<int dim>
void Xfem<dim>::write_plot ( std::vector<Point<dim>> &mesh_,
                             std::vector<double> &solution_,
                             std::string filename)
{
  filename += ".vtk";

  std::ofstream out (filename.c_str());
  out<<"# vtk DataFile Version 3.0"<<std::endl;
  out<<"#vtk This file was generated by the deal.II library"<<std::endl;
  out<<"ASCII"<<std::endl;
  out<<"DATASET UNSTRUCTURED_GRID"<<std::endl<<std::endl;

  unsigned int m=mesh_.size();
  unsigned int d=pow(2,dim);
  unsigned int n=m/d;

  out<<"POINTS "<< m <<" double"<<std::endl;

  for (unsigned int q=0; q<m; ++q)
    {
      out << mesh_[q];
      if (dim==2)
        out<<" 0"<<std::endl;
      else
        {
          std::cout<<"Only 2d case is implemented in this code"<<std::cout;
          assert(0);
        }
    }

  out<<std::endl<<"CELLS "<< n <<" " << n*(d+1) <<std::endl;

  for (unsigned int q=0; q<n; ++q)
    out<<d<<" "<<q *d<<" "<<q *d+1<<" "<<q *d+3<<" "<<q *d+2<<std::endl;

  out<<std::endl<<"CELL_TYPES" <<" "<< n <<std::endl;

  for (unsigned int q=0; q<n; ++q)
    out<<"9 ";

  out<<std::endl<<"POINT_DATA " << m <<std::endl;
  out<<"SCALARS solution_dc double 1"<<std::endl;
  out<<"LOOKUP_TABLE default"<<std::endl;

  if (m!=solution_.size())
    {
      std::cout<<"solution- and mesh-vector don't match"<<std::endl;
      assert(0);
    }

  for (unsigned int q=0; q<solution_.size(); ++q)
    out << solution_[q] << std::endl;
}

//The next two functions are the linearized level set function and its gradient. The parameters n, c are calculated during the subdivision of the cell. So, in order to use this function, one needs to make sure to call the calculate_subdivision with the current cell beforehand.

template<int dim>
double Xfem<dim>::level_set(const Point<dim> &p)
{
  return n0*p[0] + n1*p[1] - c;
}

template<int dim>
std::vector<Point<dim> > Xfem<dim>::get_intersection_points()
{
  return intersection_points;
}


template class Xfem<2>;
