// Copyright (C) 2015 by Thomas Carraro (1) and Sven Wetterauer (2)
// //
// // (1) Institute for Applied Mathematics, Heidelberg University
// //     E-mail: thomas.carraro@iwr.uni-heidelberg.de
// // (2) Institute for Applied Mathematics, Heidelberg University
// //     E-Mail: sven.wetterauer@iwr.uni-heidelberg.de
// //
// // This file is subject to the GNU Lesser General Public License
// // (LGPL) as published by the Free Software Foundation; either
// // version 2.1 of the License, or (at your option) any later
// // version. Please refer to the webpage http://www.dealii.org/
// // under the link License for the text and further information
// // on this license. You can use it, redistribute it, and/or
// // modify it under the terms of this licence.

#include <deal.II/base/function.h>
#include <deal.II/base/convergence_table.h>
#include <deal.II/base/parameter_handler.h>
#include <deal.II/base/quadrature_lib.h>
#include <deal.II/dofs/dof_handler.h>
#include <deal.II/dofs/dof_tools.h>
#include <deal.II/fe/fe_q.h>
#include <deal.II/fe/fe_nothing.h>
#include <deal.II/fe/fe_system.h>
#include <deal.II/fe/fe_values.h>
#include <deal.II/grid/tria.h>
#include <deal.II/grid/grid_generator.h>
#include <deal.II/grid/grid_tools.h>
#include <deal.II/grid/tria_boundary_lib.h>
#include <deal.II/lac/vector.h>
#include <deal.II/lac/full_matrix.h>
#include <deal.II/lac/sparse_matrix.h>
#include <deal.II/lac/dynamic_sparsity_pattern.h>
#include <deal.II/lac/solver_cg.h>
#include <deal.II/lac/precondition.h>
#include <deal.II/numerics/vector_tools.h>
#include <deal.II/numerics/matrix_tools.h>

#include "level_set.h"
#include "xfem_functions.h"
#include "xfe_values.h"

#include <ctime>

namespace XFEM_Tutorial
{
  using namespace dealii;


  double sign (double d)
  {
    if (d > 0)
      return 1;
    else if (d < 0)
      return -1;
    else
      return 0;
  }


  template <int dim>
  class WeakDiscontinuity
  {
  public:
    WeakDiscontinuity ();
    ~WeakDiscontinuity ();

    void run ();

  private:
    void setup_system ();
    void assemble_system ();
    void solve ();
    void compute_error () ;
    double exact_solution (const Point<dim> &p);
    Tensor<1,dim> exact_solution_grad (const Point<dim> &p);
    void read_param ();
    void output_convergence ();

    Triangulation<dim>    triangulation;

    hp::DoFHandler<dim>   dof_handler;
    hp::FECollection<dim> fe_collection;

    ConstraintMatrix      constraints;

    SparsityPattern       sparsity_pattern;
    SparseMatrix<double>  system_matrix;

    Vector<double>        solution;
    Vector<double>        system_rhs;

    ConvergenceTable      convergence_table;
    ParameterHandler      prm;

    Level_Set<dim>      ls;
    Xfem<dim>   xfem;

    bool      use_XFEM;
    int       n_cycles_total;
    int   n_qua_points;
    bool    activate_blending;
  };


  template <int dim>
  class RightHandSide : public Function<dim>
  {
  public:
    RightHandSide () : Function<dim>() {}

    virtual double value (const Point<dim>   &p,
                          const unsigned int  component = 0) const;
  };

  template <int dim>
  double RightHandSide<dim>::value (const Point<dim> &p,
                                    const unsigned int) const
  {
    return 1.;
  }



  template <int dim>
  class Coefficient : public Function<dim>
  {
  public:
    Coefficient () : Function<dim>() {}

    Level_Set<dim> ls;

    virtual double value (const Point<dim>   &p,
                          const unsigned int  component = 0) const;

    virtual void value_list (const std::vector<Point<dim> > &points,
                             std::vector<double>            &values,
                             const unsigned int              component = 0) const;
  };



  template <int dim>
  double Coefficient<dim>::value (const Point<dim> &p,
                                  const unsigned int) const
  {
// During the assembling we need to get the coefficient value at the quadrature points. Since the quadrature points are calculated with a linearized level set function, it is possible that quadrature points lie between the linearized and the exact interface, if these two differ too much. In this case we would get the wrong coefficient value.
// Mesh refinement would solve the problem.
// To avoid this problem on coarse meshes one could use a different quadrature rule.
// In our case, since the interface is circular and the starting mesh is fine enough this problem does not occur. We can therefore use the exact level set function to distinguish the two subdomains and assign the coefficient values.
    if (ls.level_set(p) < 0)
      return 20.;
    else
      return 1.;
  }



  template <int dim>
  void Coefficient<dim>::value_list (const std::vector<Point<dim> > &points,
                                     std::vector<double>            &values,
                                     const unsigned int              component) const
  {
    const unsigned int n_points = points.size();

    Assert (values.size() == n_points,
            ExcDimensionMismatch (values.size(), n_points));

    Assert (component == 0,
            ExcIndexRange (component, 0, 1));

    for (unsigned int i=0; i<n_points; ++i)
      {
        if (ls.level_set(points[i]) < 0)
          values[i] = 20.;
        else
          values[i] = 1.;
      }
  }


  template <int dim>
  WeakDiscontinuity<dim>::WeakDiscontinuity ()
    :
    dof_handler (triangulation)
  {
// The solution consists of two components, one for the standard part and one for the enriched part.
    fe_collection.push_back (FESystem<dim> (FE_Q<dim>(1), 1,
                                            FE_Nothing<dim>(), 1));
    fe_collection.push_back (FESystem<dim> (FE_Q<dim>(1), 1,
                                            FE_Q<dim>(1), 1));
  }



  template <int dim>
  WeakDiscontinuity<dim>::~WeakDiscontinuity ()
  {
    dof_handler.clear ();
  }



// The active_fe_index determines which kind of finite elements are used for a cell. In the setup_system we need to assign to every cell the correct active_fe_index.
  template <int dim>
  void WeakDiscontinuity<dim>::setup_system ()
  {
    for (typename hp::DoFHandler<dim>::cell_iterator cell=dof_handler.begin_active();
         cell!=dof_handler.end(); ++cell)
      cell->set_active_fe_index(0);

    if (use_XFEM)
      for (typename hp::DoFHandler<dim>::cell_iterator
           cell = dof_handler.begin_active();
           cell != dof_handler.end(); ++cell)
        if (xfem.interface_intersects_cell(cell))
          {
// All intersected cells have an active_fe_index equal to 1, because the solution in these cells consists of two nonzero components.
            cell->set_active_fe_index(1);
            if (activate_blending)
              {
// All cells surrounding an intersected cell are blending cells. Therefore, we not only need the neighboring cells, but also the diagonal neighbors of the current cell. This is done with the help of the find_all_neighbors function of the xfem-class.
                std::vector<typename hp::DoFHandler<dim>::active_cell_iterator> adjacent_cells
                  =xfem.find_all_neighbors(cell);
                for (unsigned int i=0; i<adjacent_cells.size(); ++i)
                  {
                    adjacent_cells[i]->set_active_fe_index(1);
                  }
              }
          }

    dof_handler.distribute_dofs (fe_collection);

    solution.reinit (dof_handler.n_dofs());
    system_rhs.reinit (dof_handler.n_dofs());

    constraints.clear ();
    constraints.close();

    DynamicSparsityPattern d_sparsity(dof_handler.n_dofs());
    DoFTools::make_sparsity_pattern (dof_handler, d_sparsity);

    constraints.condense (d_sparsity);

    sparsity_pattern.copy_from(d_sparsity);

    system_matrix.reinit (sparsity_pattern);
  }


  template <int dim>
  void WeakDiscontinuity<dim>::assemble_system ()
  {
    const QGauss<dim>  quadrature_formula(n_qua_points);

    FEValues<dim> plain_fe_values (fe_collection[0], quadrature_formula,
                                   update_values    |  update_gradients |
                                   update_quadrature_points  |  update_JxW_values);

    const unsigned int   n_q_points    = quadrature_formula.size();

    FullMatrix<double>   cell_matrix;
    Vector<double>       cell_rhs;

    std::vector<types::global_dof_index> local_dof_indices;

    const Coefficient<dim> coefficient;
    std::vector<double>    coefficient_values (n_q_points);

    typename hp::DoFHandler<dim>::active_cell_iterator
    cell = dof_handler.begin_active(),
    endc = dof_handler.end();

    for (; cell!=endc; ++cell)
      {
        const unsigned int dofs_per_cell = cell->get_fe().dofs_per_cell;
        cell_matrix.reinit (dofs_per_cell, dofs_per_cell);
        cell_rhs.reinit (dofs_per_cell);

        cell_matrix = 0;
        cell_rhs = 0;
// From here on, the assemble_system differs from the deal.II tutorial programs. We need to distinguish between standard (active_fe_index==0) and cut cells (active_fe_index==1). For the standard cells we compute the local cell-matrix in the same way as in the introductory tutorials, like step-3.

        if (cell->active_fe_index() == 0)
          {
            plain_fe_values.reinit (cell);

            coefficient_values.resize (plain_fe_values.n_quadrature_points);
            coefficient.value_list (plain_fe_values.get_quadrature_points(),
                                    coefficient_values);

            for (unsigned int q_point=0; q_point<n_q_points; ++q_point)
              for (unsigned int i=0; i<dofs_per_cell; ++i)
                {
                  for (unsigned int j=0; j<dofs_per_cell; ++j)
                    cell_matrix(i,j) +=( coefficient_values[q_point]
                                         * plain_fe_values.shape_grad(i,q_point)
                                         * plain_fe_values.shape_grad(j,q_point)
                                         * plain_fe_values.JxW(q_point));


                  cell_rhs(i) +=( plain_fe_values.shape_value(i,q_point)
                                  * 1.0
                                  * plain_fe_values.JxW(q_point));
                }
          }

// If the cell is not a standard cell it can either be intersected by the interface or it is a blending cell. Both types have active_fe_index==1. We do not need to distinguish between these two types, as this is done in the special FEValues-class used here, called XFEValues_weak.
        else
          {
            Assert (cell->active_fe_index() == 1, ExcInternalError());

// The additional shape functions have a discontinuity along the interface, which is inside a cell, meaning that standard quadrature rules do not work optimally as they require a certain regularity inside the cell. We define a new quadrature rule by subdividing the cell along the interface and applying a standard quadrature rule on each subcell. As this quadrature formula might be different for every cell, we have to call the constructor for the special FEValues class for each cell.

            XFEValues_weak<dim> xfem_fe_values (fe_collection[1],
                                                xfem.compute_quadrature(quadrature_formula, cell),
                                                update_values    |  update_gradients |
                                                update_JxW_values );

            xfem_fe_values.reinit (cell);

            coefficient_values.resize (xfem_fe_values.n_quadrature_points);
            coefficient.value_list (xfem_fe_values.get_quadrature_points(),
                                    coefficient_values);

            for (unsigned int q_point=0; q_point<xfem_fe_values.n_quadrature_points; ++q_point)
              for (unsigned int i=0; i<dofs_per_cell; ++i)
                {
                  for (unsigned int j=0; j<dofs_per_cell; ++j)
                    cell_matrix(i,j) +=( coefficient_values[q_point]
                                         * xfem_fe_values.shape_grad(i,q_point)
                                         * xfem_fe_values.shape_grad(j,q_point)
                                         * xfem_fe_values.JxW(q_point));
                  cell_rhs(i) +=( xfem_fe_values.shape_value(i,q_point)
                                  * 1.0
                                  * xfem_fe_values.JxW(q_point));
                }
          }
//The remainder of this function is the same as in the tutorials.
        local_dof_indices.resize (dofs_per_cell);
        cell->get_dof_indices (local_dof_indices);
        constraints.distribute_local_to_global (cell_matrix, cell_rhs,
                                                local_dof_indices,
                                                system_matrix, system_rhs);
      }

    std::map<types::global_dof_index,double> boundary_values;
    VectorTools::interpolate_boundary_values (dof_handler,
                                              0,
                                              ZeroFunction<dim>(2),
                                              boundary_values);
    MatrixTools::apply_boundary_values (boundary_values,
                                        system_matrix,
                                        solution,
                                        system_rhs);
  }


  template <int dim>
  void WeakDiscontinuity<dim>::solve ()
  {
    SolverControl           solver_control (10000, 1e-12);
    SolverCG<>              solver (solver_control);

    PreconditionSSOR<> preconditioner;
    preconditioner.initialize(system_matrix, 1.2);

    solver.solve (system_matrix, solution, system_rhs,
                  preconditioner);

    constraints.distribute (solution);
  }


// The compute_error function is self explaining. We calculate the difference of the computed solution and the exact solution, square it and integrate it over the cell.
// The only step that needs notice is the evaluation of the computed solution, because we need to distinguish between standard and intersected cells. The shape functions in these two cases are the same as in the assemble_system.

  template <int dim>
  void WeakDiscontinuity<dim>::compute_error ()
  {
    const QGauss<dim>  quadrature_formula(n_qua_points);

    FEValues<dim> plain_fe_values (fe_collection[0], quadrature_formula,
                                   update_values    |  update_gradients |
                                   update_q_points  |  update_JxW_values);

    double l2_error_square = 0;
    double energy_error_square = 0;

    typename hp::DoFHandler<dim>::active_cell_iterator
    cell = dof_handler.begin_active(),
    endc = dof_handler.end();

    for (; cell!=endc; ++cell)
      {
        double local_l2_error=0;
        Tensor<1,dim> local_energy_error;

        const unsigned int dofs_per_cell=cell->get_fe().dofs_per_cell;

        if (cell->active_fe_index()==0)
          {
            plain_fe_values.reinit(cell);
            Vector<double> dof_values(dofs_per_cell);
            cell->get_dof_values(solution,dof_values);
            for (unsigned int q=0; q<plain_fe_values.n_quadrature_points; ++q)
              {
                local_l2_error=0;
                local_energy_error[0]=0;
                local_energy_error[1]=0;
                for (unsigned int i=0; i<dofs_per_cell; ++i)
                  {
                    local_l2_error += dof_values[i]
                                      * plain_fe_values.shape_value(i,q);
                    local_energy_error += dof_values[i]
                                          * plain_fe_values.shape_grad(i,q);
                  }
                local_l2_error -= exact_solution(plain_fe_values.quadrature_point(q));
                local_energy_error -= exact_solution_grad(plain_fe_values.quadrature_point(q));
                l2_error_square += local_l2_error * local_l2_error * plain_fe_values.JxW(q);
                energy_error_square += local_energy_error * local_energy_error * plain_fe_values.JxW(q);
              }
          }
        else
          {
            XFEValues_weak<dim> xfem_fe_values (fe_collection[1], xfem.compute_quadrature(quadrature_formula, cell),
                                                update_values    |  update_gradients |
                                                update_JxW_values);
            xfem_fe_values.reinit(cell);
            Vector<double> dof_values(dofs_per_cell);
            cell->get_dof_values(solution,dof_values);
            for (unsigned int q=0; q<xfem_fe_values.n_quadrature_points; ++q)
              {
                local_l2_error=0;
                local_energy_error[0]=0;
                local_energy_error[1]=0;
                for (unsigned int i=0; i<dofs_per_cell; ++i)
                  {
                    local_l2_error += dof_values[i]
                                      * xfem_fe_values.shape_value(i,q);
                    local_energy_error += dof_values[i]
                                          * xfem_fe_values.shape_grad(i,q);
                  }

                local_l2_error-=exact_solution(xfem_fe_values.quadrature_point(q));
                local_energy_error -= exact_solution_grad(xfem_fe_values.quadrature_point(q));
                l2_error_square += local_l2_error * local_l2_error * xfem_fe_values.JxW(q);
                energy_error_square += local_energy_error * local_energy_error * xfem_fe_values.JxW(q);
              }
          }
      }

    std::cout << "   L2 error = " << std::sqrt (l2_error_square)
              << std::endl;
    std::cout << "   energy error = "<<std::sqrt(energy_error_square)<<std::endl;
    const double L2_error=std::sqrt(l2_error_square);
    const double energy_error=std::sqrt(energy_error_square);
    convergence_table.add_value("L2", L2_error);
    convergence_table.add_value("Energy",energy_error);
  }


  template <int dim>
  double WeakDiscontinuity<dim>::exact_solution (const Point<dim> &p)
  {
    if (ls.level_set(p)<0)
      return 1./20 * (-1./4*p.square() + 61./16);
    else
      return 1./4 * (1-p.square());
  }

  template <int dim>
  Tensor<1,dim> WeakDiscontinuity<dim>::exact_solution_grad (const Point<dim> &p)
  {
    Tensor<1,dim> result;
    if (ls.level_set(p) < 0)
      {
        result[0]=1./20. * (-1./2. * p(0));
        result[1]=1./20. * (-1./2. * p(1));
      }
    else
      {
        result[0]=-1./2.*p(0);
        result[1]=-1./2.*p(1);
      }
    return result;
  }


  template <int dim>
  void WeakDiscontinuity<dim>::output_convergence()
  {
    convergence_table.set_precision("L2", 3);
    convergence_table.set_scientific("L2", true);
    convergence_table
    .evaluate_convergence_rates("L2", ConvergenceTable::reduction_rate_log2);
    convergence_table.set_precision("Energy", 3);
    convergence_table.set_scientific("Energy", true);
    convergence_table
    .evaluate_convergence_rates("Energy", ConvergenceTable::reduction_rate_log2);
    std::cout << std::endl;
    convergence_table.write_text(std::cout);
  }


  template <int dim>
  void WeakDiscontinuity<dim>::read_param()
  {
    prm.declare_entry ("Using XFEM", "true",
                       Patterns::Bool(),
                       "A parameter determining, if XFEM or FEM are used");
    prm.declare_entry("Number of Cycles", "6",
                      Patterns::Integer(),
                      "Integer to define the number of Refinement cycles");
    prm.declare_entry("q_points", "3",
                      Patterns::Integer(),
                      "Number of quadrature points used for every integration");
    prm.declare_entry ("blending", "true",
                       Patterns::Bool(),
                       "A parameter to activate the additional blending cells");

    prm.read_input("param.prm");

    use_XFEM=prm.get_bool("Using XFEM");
    n_cycles_total=prm.get_integer("Number of Cycles");
    n_qua_points=prm.get_integer("q_points");
    activate_blending=prm.get_bool("blending");
  }


  template <int dim>
  void WeakDiscontinuity<dim>::run ()
  {
    read_param ();

    for (unsigned int cycle=0; cycle<n_cycles_total; ++cycle)
      {
        std::cout << "Cycle " << cycle << ':' << std::endl;

        if (cycle == 0)
          {
            GridGenerator::hyper_ball (triangulation);
            static const HyperBallBoundary<dim> boundary;
            triangulation.set_boundary (0, boundary);
            triangulation.refine_global (2);
          }
        else
          triangulation.refine_global (1);

        std::cout << "   Number of active cells:       "
                  << triangulation.n_active_cells()
                  << std::endl;

        setup_system ();

        std::cout << "   Number of degrees of freedom: "
                  << dof_handler.n_dofs()
                  << std::endl;

        assemble_system ();
        solve ();
        compute_error();
        xfem.plot_vtk_weak(dof_handler,fe_collection,solution,cycle);
      }
    output_convergence();
  }
}


int main ()
{
  try
    {
      using namespace dealii;
      using namespace XFEM_Tutorial;

      deallog.depth_console (0);

      WeakDiscontinuity<2> weak_2d;
      weak_2d.run ();
    }
  catch (std::exception &exc)
    {
      std::cerr << std::endl << std::endl
                << "----------------------------------------------------"
                << std::endl;
      std::cerr << "Exception on processing: " << std::endl
                << exc.what() << std::endl
                << "Aborting!" << std::endl
                << "----------------------------------------------------"
                << std::endl;

      return 1;
    }
  catch (...)
    {
      std::cerr << std::endl << std::endl
                << "----------------------------------------------------"
                << std::endl;
      std::cerr << "Unknown exception!" << std::endl
                << "Aborting!" << std::endl
                << "----------------------------------------------------"
                << std::endl;
      return 1;
    }
  return 0;
}
