#include <cassert>
#include <cmath>
#include <iostream>
#include "quadrature.hpp"


using namespace pardg;



// 3d quadrature rules for the reference
// tetrahedron [(0,0,0), (1,0,0), (0,1,0), (0,0,1)]
//
// naming convention of quad3d_x: x denotes the order of the formula
//                                i.e. the maximum degree of polynimials
//                                that is integrated exact by the formula.
//
// exception quad3d_0: this is a dummy that has no points



namespace pardg
{
// return a quadrature formula that has at least degree minimum_degree
template<>
const Quadrature3d& Quadrature3d::quadrature(int minimum_degree)
{
  if (minimum_degree < 0) return quad3d_0;

  switch (minimum_degree){
  case 0:
  case 1: return quad3d_1; break;
  case 2: return quad3d_2; break;
  case 3: return quad3d_3; break;
  case 4: return quad3d_4; break;
  case 5: return quad3d_5; break;
  case 6: return quad3d_6; break;
  case 7: return quad3d_7; break;
  case 8: return quad3d_8; break;
  case 9: return quad3d_9; break;
  case 10:
  case 11: return quad3d_11; break;
  default:
    {
      std::cerr << "Quadrature3d::quadrature: order " << minimum_degree << " not avaiable" << std::endl;
      assert(0);
      abort();
    }
  }

  return quad3d_0;
}



template<>
void Quadrature3d::check() const
{
  double total_error = 0.0;

  for(int i=0; i<=degree; i++){
    for(int j=0; j<=i; j++){
      for(int l=0; l<=j; l++){
	int i_j = i-j;
	int j_l = j-l;

	double sum = 0.0;
	for(int r=0; r<nop; r++){
	  double x_i_j = 1.0; // x^{i-j}
	  double y_j_l = 1.0; // y^{j-l}
	  double z_l = 1.0;    // z^l
	  for(int k=1; k<=i_j; k++) x_i_j *= x(r)[0];
	  for(int k=1; k<=j_l; k++) y_j_l *= x(r)[1];
	  for(int k=1; k<=l; k++) z_l *= x(r)[2];
	  sum += w(r) * x_i_j * y_j_l * z_l;
	}

	long long a = 1;
	long long b = 1;
	for(int k=1; k<=i_j; k++) a *= k;
	for(int k=1; k<=j_l; k++) a *= k;
	for(int k=1; k<=l; k++) a *= k;
	for(int k=1; k<=i+3; k++) b *= k;

	double error = fabs(sum - (double)a / (double)b);
	total_error += error;

	std::cout << i_j << "  "
		  << j_l << "  "
		  << l << "  "
		  << "     error: " << error << "  "
		  << std::endl;
      }
    }
  }

  std::cout << "total error: " << total_error << std::endl;
}

} // namespace pardg


// dummy
static const double quad3d_0_x[][4] = {{}};
const Quadrature3d pardg::quad3d_0(0, 0, quad3d_0_x);




static const double quad3d_1_x[][4] =
  {{0.25, 0.25, 0.25,   1.0/6.0}
  };
const Quadrature3d pardg::quad3d_1(1, 1, quad3d_1_x);



// Approximate calculation of multiple integrals / A. H. Stroud.
// Englewood Cliffs, NJ : Prentice-Hall, c 1971. - XIII,
// 431 S. : graph. Darst.;
// (engl.)(Prentice-Hall series in automatic computation)
// ISBN 0-13-043893-6
//
// page 307
//
static const double quad3d_2_x[][4] =
  {{(5.0-sqrt(5.0))/20.0, (5.0-sqrt(5.0))/20.0, (5.0-sqrt(5.0))/20.0,
    1.0/24.0},
   {(5.0+3.0*sqrt(5.0))/20.0, (5.0-sqrt(5.0))/20.0, (5.0-sqrt(5.0))/20.0,
    1.0/24.0},
   {(5.0-sqrt(5.0))/20.0, (5.0+3.0*sqrt(5.0))/20.0, (5.0-sqrt(5.0))/20.0,
    1.0/24.0},
   {(5.0-sqrt(5.0))/20.0, (5.0-sqrt(5.0))/20.0, (5.0+3.0*sqrt(5.0))/20.0,
    1.0/24.0}
  };
const Quadrature3d pardg::quad3d_2(4, 2, quad3d_2_x);




// Approximate calculation of multiple integrals / A. H. Stroud.
// Englewood Cliffs, NJ : Prentice-Hall, c 1971. - XIII,
// 431 S. : graph. Darst.;
// (engl.)(Prentice-Hall series in automatic computation)
// ISBN 0-13-043893-6
//
// page 308
//
static const double quad3d_3_x[][4] =
  {{0.25, 0.25, 0.25,           -2.0/15.0}, // w0
   {1.0/6.0, 1.0/6.0, 1.0/6.0,  3.0/40.0}, // w1
   {1.0/2.0, 1.0/6.0, 1.0/6.0,  3.0/40.0},
   {1.0/6.0, 1.0/2.0, 1.0/6.0,  3.0/40.0},
   {1.0/6.0, 1.0/6.0, 1.0/2.0,  3.0/40.0}
  };
const Quadrature3d pardg::quad3d_3(5, 3, quad3d_3_x);



// P. Keast,  Moderate-degree tetrahedral quadrature formulas,
// Comput. Methods Appl. Mech. Engrg.  55 (1986), 339--348.
//
// http://www.cs.kuleuven.ac.be/~nines/research/ecf/mtables.html
// login=encyclop, passwd=stroud71
//
// note: has negative weights
static const double quad3d_4_x[][4] =
  {{0.25, 0.25, 0.25,
    -0.013155555555555555}, // w0
   {0.785714285714285716, 0.071428571428571428 ,0.071428571428571428,
    7.6222222222222222e-3}, // w1
   {0.071428571428571428, 0.785714285714285716 ,0.071428571428571428,
    7.6222222222222222e-3},
   {0.071428571428571428, 0.071428571428571428 ,0.785714285714285716,
    7.6222222222222222e-3},
   {0.071428571428571428, 0.071428571428571428 ,0.071428571428571428,
    7.6222222222222222e-3},
   {0.39940357616679920, 0.39940357616679920 ,0.10059642383320079,
    0.024888888888888888}, // w2
   {0.39940357616679920, 0.10059642383320079 ,0.39940357616679920,
    0.024888888888888888},
   {0.39940357616679920, 0.10059642383320079 ,0.10059642383320079,
    0.024888888888888888},
   {0.10059642383320079, 0.39940357616679920 ,0.39940357616679920,
    0.024888888888888888},
   {0.10059642383320079, 0.39940357616679920 ,0.10059642383320079,
    0.024888888888888888},
   {0.10059642383320079, 0.10059642383320079 ,0.39940357616679920,
    0.024888888888888888}
  };
const Quadrature3d pardg::quad3d_4(11, 4, quad3d_4_x);




// "Quadrature on Simplices of Arbitrary Dimension"
// Noel J. Walkington
// Scientific Report 2000
// 00-CNA-023 (Center for Nonlinear Analysis)
static const double quad3d_5_x[][4] =
  {{0.31088591926330060980, 0.31088591926330060980, 0.31088591926330060980,
    0.018781320953002641800}, // w0
   {0.06734224221009817060, 0.31088591926330060980, 0.31088591926330060980,
    0.018781320953002641800},
   {0.31088591926330060980, 0.06734224221009817060, 0.31088591926330060980,
    0.018781320953002641800},
   {0.31088591926330060980, 0.31088591926330060980, 0.06734224221009817060,
    0.018781320953002641800},
   {0.092735250310891226402, 0.092735250310891226402, 0.092735250310891226402,
    0.012248840519393658257}, // w1
   {0.721794249067326320794, 0.092735250310891226402, 0.092735250310891226402,
    0.012248840519393658257},
   {0.092735250310891226402, 0.721794249067326320794, 0.092735250310891226402,
    0.012248840519393658257},
   {0.092735250310891226402, 0.092735250310891226402, 0.721794249067326320794,
    0.012248840519393658257},
   {0.045503704125649649492, 0.045503704125649649492, 0.454496295874350350508,
    0.0070910034628469110730}, // w2
   {0.045503704125649649492, 0.454496295874350350508, 0.045503704125649649492,
    0.0070910034628469110730},
   {0.454496295874350350508, 0.045503704125649649492, 0.045503704125649649492,
    0.0070910034628469110730},
   {0.454496295874350350508, 0.454496295874350350508, 0.045503704125649649492,
    0.0070910034628469110730},
   {0.454496295874350350508, 0.045503704125649649492, 0.454496295874350350508,
    0.0070910034628469110730},
   {0.045503704125649649492, 0.454496295874350350508, 0.454496295874350350508,
    0.0070910034628469110730}
  };
const Quadrature3d pardg::quad3d_5(14, 5, quad3d_5_x);





// stolen from ALBERTA sources
static const double quad3d_5b_x[][4] =
  {{0.250000000000000, 0.250000000000000, 0.250000000000000,
    0.118518518518519/6.0}, // w0
   {0.091971078052723, 0.091971078052723, 0.091971078052723,
    0.071937083779019/6.0}, // w1
   {0.724086765841831, 0.091971078052723, 0.091971078052723,
    0.071937083779019/6.0},
   {0.091971078052723, 0.724086765841831, 0.091971078052723,
    0.071937083779019/6.0},
   {0.091971078052723, 0.091971078052723, 0.724086765841831,
    0.071937083779019/6.0},
   {0.319793627829630, 0.319793627829630, 0.319793627829630,
    0.069068207226272/6.0}, // w2
   {0.040619116511110, 0.319793627829630, 0.319793627829630,
    0.069068207226272/6.0},
   {0.319793627829630, 0.040619116511110, 0.319793627829630,
    0.069068207226272/6.0},
   {0.319793627829630, 0.319793627829630, 0.040619116511110,
    0.069068207226272/6.0},
   {0.443649167310371, 0.056350832689629, 0.056350832689629,
    0.052910052910053/6.0}, // w3
   {0.056350832689629, 0.443649167310371, 0.056350832689629,
    0.052910052910053/6.0},
   {0.056350832689629, 0.056350832689629, 0.443649167310371,
    0.052910052910053/6.0},
   {0.443649167310371, 0.056350832689629, 0.443649167310371,
    0.052910052910053/6.0},
   {0.443649167310371, 0.443649167310371, 0.056350832689629,
    0.052910052910053/6.0},
   {0.056350832689629, 0.443649167310371, 0.443649167310371,
    0.052910052910053/6.0}
  };
const Quadrature3d pardg::quad3d_5b(15, 5, quad3d_5b_x);




// P. Keast,  Moderate-degree tetrahedral quadrature formulas,
// Comput. Methods Appl. Mech. Engrg.  55 (1986), 339--348.
//
// http://www.cs.kuleuven.ac.be/~nines/research/ecf/mtables.html
// login=encyclop, passwd=stroud71
//
static const double quad3d_6_x[][4] =
  {{0.35619138622254394, 0.21460287125915202 ,0.21460287125915202,
    6.6537917096945820e-3}, // w0
   {0.21460287125915202, 0.35619138622254394 ,0.21460287125915202,
    6.6537917096945820e-3},
   {0.21460287125915202, 0.21460287125915202 ,0.35619138622254394,
    6.6537917096945820e-3},
   {0.21460287125915202, 0.21460287125915202 ,0.21460287125915202,
    6.6537917096945820e-3},
   {0.877978124396165941, 0.040673958534611353 ,0.040673958534611353,
    1.6795351758867738e-3}, // w1
   {0.040673958534611353, 0.877978124396165941 ,0.040673958534611353,
    1.6795351758867738e-3},
   {0.040673958534611353, 0.040673958534611353 ,0.877978124396165941,
    1.6795351758867738e-3},
   {0.040673958534611353, 0.040673958534611353 ,0.040673958534611353,
    1.6795351758867738e-3},
   {0.03298632957317347, 0.32233789014227551, 0.32233789014227551,
    9.2261969239424536e-3}, // w2
   {0.32233789014227551, 0.03298632957317347, 0.32233789014227551,
    9.2261969239424536e-3},
   {0.32233789014227551, 0.32233789014227551, 0.03298632957317347,
    9.2261969239424536e-3},
   {0.32233789014227551, 0.32233789014227551, 0.32233789014227551,
    9.2261969239424536e-3},
   {0.603005664791649150, 0.26967233145831580, 0.063661001875017525,
    8.0357142857142857e-3}, // w3
   {0.603005664791649150, 0.063661001875017525, 0.26967233145831580,
    8.0357142857142857e-3},
   {0.603005664791649150, 0.063661001875017525, 0.063661001875017525,
    8.0357142857142857e-3},
   {0.26967233145831580, 0.603005664791649150, 0.063661001875017525,
    8.0357142857142857e-3},
   {0.26967233145831580, 0.063661001875017525, 0.603005664791649150,
    8.0357142857142857e-3},
   {0.26967233145831580, 0.063661001875017525, 0.063661001875017525,
    8.0357142857142857e-3},
   {0.063661001875017525, 0.603005664791649150, 0.26967233145831580,
    8.0357142857142857e-3},
   {0.063661001875017525, 0.603005664791649150, 0.063661001875017525,
    8.0357142857142857e-3},
   {0.063661001875017525, 0.26967233145831580, 0.603005664791649150,
    8.0357142857142857e-3},
   {0.063661001875017525, 0.26967233145831580, 0.063661001875017525,
    8.0357142857142857e-3},
   {0.063661001875017525, 0.063661001875017525, 0.603005664791649150,
    8.0357142857142857e-3},
   {0.063661001875017525, 0.063661001875017525, 0.26967233145831580,
    8.0357142857142857e-3},
  };
const Quadrature3d pardg::quad3d_6(24, 6, quad3d_6_x);




// "Quadrature on Simplices of Arbitrary Dimension"
// Noel J. Walkington
// Scientific Report 2000
// 00-CNA-023 (Center for Nonlinear Analysis)
static const double quad3d_7b_x[][4] =
  {{0.25, 0.25, 0.25,   -8.0/945.0}, // w0
   {1.0/6.0, 1.0/6.0, 1.0/6.0,   243.0/4480.0}, // w1
   {0.5, 1.0/6.0, 1.0/6.0,       243.0/4480.0},
   {1.0/6.0, 0.5, 1.0/6.0,       243.0/4480.0},
   {1.0/6.0, 1.0/6.0, 0.5,       243.0/4480.0},
   {0.125, 0.125, 0.125,         -256.0/2835.0}, // w2
   {0.625, 0.125, 0.125,         -256.0/2835.0},
   {0.125, 0.625, 0.125,         -256.0/2835.0},
   {0.125, 0.125, 0.625,         -256.0/2835.0},
   {0.125, 0.125, 0.375,         -256.0/2835.0}, // w2
   {0.125, 0.375, 0.125,         -256.0/2835.0},
   {0.375, 0.125, 0.125,         -256.0/2835.0},
   {0.375, 0.375, 0.125,         -256.0/2835.0},
   {0.375, 0.125, 0.375,         -256.0/2835.0},
   {0.125, 0.375, 0.375,         -256.0/2835.0},
   {0.1, 0.1, 0.1,               3125.0/72576.0}, // w3
   {0.7, 0.1, 0.1,               3125.0/72576.0},
   {0.1, 0.7, 0.1,               3125.0/72576.0},
   {0.1, 0.1, 0.7,               3125.0/72576.0},
   {0.1, 0.3, 0.5,               3125.0/72576.0}, // w3
   {0.1, 0.5, 0.3,               3125.0/72576.0},
   {0.3, 0.1, 0.5,               3125.0/72576.0},
   {0.3, 0.5, 0.1,               3125.0/72576.0},
   {0.5, 0.1, 0.3,               3125.0/72576.0},
   {0.5, 0.3, 0.1,               3125.0/72576.0},
   {0.1, 0.1, 0.3,               3125.0/72576.0},
   {0.1, 0.3, 0.1,               3125.0/72576.0},
   {0.3, 0.1, 0.1,               3125.0/72576.0},
   {0.1, 0.1, 0.5,               3125.0/72576.0},
   {0.1, 0.5, 0.1,               3125.0/72576.0},
   {0.5, 0.1, 0.1,               3125.0/72576.0},
   {0.3, 0.3, 0.3,               3125.0/72576.0}, // w3
   {0.1, 0.3, 0.3,               3125.0/72576.0},
   {0.3, 0.1, 0.3,               3125.0/72576.0},
   {0.3, 0.3, 0.1,               3125.0/72576.0}
  };
static const double quad3d_7b_w[] =
  {-8.0/945.0, // w0
   243.0/4480.0, // w1
   243.0/4480.0,
   243.0/4480.0,
   243.0/4480.0,
   -256.0/2835.0, // w2
   -256.0/2835.0,
   -256.0/2835.0,
   -256.0/2835.0,
   -256.0/2835.0, // w2
   -256.0/2835.0,
   -256.0/2835.0,
   -256.0/2835.0,
   -256.0/2835.0,
   -256.0/2835.0,
   3125.0/72576.0, // w3
   3125.0/72576.0,
   3125.0/72576.0,
   3125.0/72576.0,
   3125.0/72576.0, // w3
   3125.0/72576.0,
   3125.0/72576.0,
   3125.0/72576.0,
   3125.0/72576.0,
   3125.0/72576.0,
   3125.0/72576.0,
   3125.0/72576.0,
   3125.0/72576.0,
   3125.0/72576.0,
   3125.0/72576.0,
   3125.0/72576.0,
   3125.0/72576.0, // w3
   3125.0/72576.0,
   3125.0/72576.0,
   3125.0/72576.0
  };
const Quadrature3d pardg::quad3d_7b(35, 7, quad3d_7b_x);



// P. Keast,
// Moderate-degree tetrahedral quadrature formulas,
// Comput. Methods Appl. Mech. Engrg.  55 (1986), 339--348.
//
// http://www.cs.kuleuven.ac.be/~nines/research/ecf/mtables.html
// login=encyclop, passwd=stroud71
//
// note: has negative weights
static const double quad3d_7_x[][4] =
  {{0.25, 0.25, 0.25,   0.018264223466108820}, // w0
   {0.5, 0.5, 0.0,      9.7001763668430335e-4}, // w1
   {0.5, 0.0, 0.5,      9.7001763668430335e-4},
   {0.5, 0.0, 0.0,      9.7001763668430335e-4},
   {0.0, 0.5, 0.5,      9.7001763668430335e-4},
   {0.0, 0.5, 0.0,      9.7001763668430335e-4},
   {0.0, 0.0, 0.5,      9.7001763668430335e-4},
   {0.078213192330318064, 0.078213192330318064, 0.078213192330318064,
    0.010599941524413686}, // w2
   {0.765360423009045808, 0.078213192330318064, 0.078213192330318064,
    0.010599941524413686},
   {0.078213192330318064, 0.765360423009045808, 0.078213192330318064,
    0.010599941524413686},
   {0.078213192330318064, 0.078213192330318064, 0.765360423009045808,
    0.010599941524413686},
   {0.12184321666390517, 0.12184321666390517, 0.12184321666390517,
    -0.062517740114331851}, // w3
   {0.63447035000828449, 0.12184321666390517, 0.12184321666390517,
    -0.062517740114331851},
   {0.12184321666390517, 0.63447035000828449, 0.12184321666390517,
    -0.062517740114331851},
   {0.12184321666390517, 0.12184321666390517, 0.63447035000828449,
    -0.062517740114331851},
   {0.33253916444642062, 0.33253916444642062, 0.33253916444642062,
    4.8914252630734993e-3}, // w4
   {0.00238250666073814, 0.33253916444642062, 0.33253916444642062,
    4.8914252630734993e-3},
   {0.33253916444642062, 0.00238250666073814, 0.33253916444642062,
    4.8914252630734993e-3},
   {0.33253916444642062, 0.33253916444642062, 0.00238250666073814,
    4.8914252630734993e-3},
   {0.6, 0.2, 0.1,   0.027557319223985890}, // w5
   {0.6, 0.1, 0.2,   0.027557319223985890},
   {0.6, 0.1, 0.1,   0.027557319223985890},
   {0.2, 0.6, 0.1,   0.027557319223985890},
   {0.2, 0.1, 0.6,   0.027557319223985890},
   {0.2, 0.1, 0.1,   0.027557319223985890},
   {0.1, 0.6, 0.2,   0.027557319223985890},
   {0.1, 0.6, 0.1,   0.027557319223985890},
   {0.1, 0.2, 0.6,   0.027557319223985890},
   {0.1, 0.2, 0.1,   0.027557319223985890},
   {0.1, 0.1, 0.6,   0.027557319223985890},
   {0.1, 0.1, 0.2,   0.027557319223985890}
  };
const Quadrature3d pardg::quad3d_7(31, 7, quad3d_7_x);




// M. Beckers and A. Haegemans,
// The construction of cubature formulae for the tetrahedron,
// Report TW 128, Dept. of Computer Science, K.U. Leuven, 1990.
//
// M. Beckers,  Numerical integration in high dimensions,
// Ph.D. thesis, Katholieke Universiteit Leuven, 1992.
//
// http://www.cs.kuleuven.ac.be/~nines/research/ecf/mtables.html
// login=encyclop, passwd=stroud71
//
// note: has negative weights
static const double quad3d_8_x[][4] =
  {{0.25, 0.25, 0.25,
    -0.020500188658639915}, // w0
   {0.20682993161067320, 0.20682993161067320, 0.20682993161067320,
    0.014250305822866901}, // w1
   {0.37951020516798040, 0.20682993161067320, 0.20682993161067320,
    0.014250305822866901},
   {0.20682993161067320, 0.37951020516798040, 0.20682993161067320,
    0.014250305822866901},
   {0.20682993161067320, 0.20682993161067320, 0.37951020516798040,
    0.014250305822866901},
   {0.082103588310546723, 0.082103588310546723, 0.082103588310546723,
    1.9670333131339009e-3}, // w2
   {0.753689235068359831, 0.082103588310546723, 0.082103588310546723,
    1.9670333131339009e-3},
   {0.082103588310546723, 0.753689235068359831, 0.082103588310546723,
    1.9670333131339009e-3},
   {0.082103588310546723, 0.082103588310546723, 0.753689235068359831,
    1.9670333131339009e-3},
   {5.7819505051979972e-3, 5.7819505051979972e-3, 5.7819505051979972e-3,
    1.6983410909288737e-4}, // w3
   {0.98265414848440600840, 5.7819505051979972e-3, 5.7819505051979972e-3,
    1.6983410909288737e-4},
   {5.7819505051979972e-3, 0.98265414848440600840, 5.7819505051979972e-3,
    1.6983410909288737e-4},
   {5.7819505051979972e-3, 5.7819505051979972e-3, 0.98265414848440600840,
    1.6983410909288737e-4},
   {0.050532740018894224, 0.050532740018894224, 0.44946725998110577,
    4.5796838244672818e-3}, // w4
   {0.050532740018894224, 0.44946725998110577, 0.050532740018894224,
    4.5796838244672818e-3},
   {0.050532740018894224, 0.44946725998110577, 0.44946725998110577,
    4.5796838244672818e-3},
   {0.44946725998110577, 0.44946725998110577, 0.050532740018894224,
    4.5796838244672818e-3},
   {0.44946725998110577, 0.050532740018894224, 0.44946725998110577,
    4.5796838244672818e-3},
   {0.44946725998110577, 0.050532740018894224, 0.050532740018894224,
    4.5796838244672818e-3},
   {0.506227344977843697, 0.035639582788534043 ,0.22906653611681113,
    5.7044858086819185e-3}, // w5
   {0.506227344977843697, 0.22906653611681113 ,0.035639582788534043,
    5.7044858086819185e-3},
   {0.506227344977843697, 0.22906653611681113 ,0.22906653611681113,
    5.7044858086819185e-3},
   {0.035639582788534043, 0.506227344977843697 ,0.22906653611681113,
    5.7044858086819185e-3},
   {0.035639582788534043, 0.22906653611681113 ,0.506227344977843697,
    5.7044858086819185e-3},
   {0.035639582788534043, 0.22906653611681113 ,0.22906653611681113,
    5.7044858086819185e-3},
   {0.22906653611681113, 0.506227344977843697 ,0.035639582788534043,
    5.7044858086819185e-3},
   {0.22906653611681113, 0.506227344977843697 ,0.22906653611681113,
    5.7044858086819185e-3},
   {0.22906653611681113, 0.035639582788534043 ,0.506227344977843697,
    5.7044858086819185e-3},
   {0.22906653611681113, 0.035639582788534043 ,0.22906653611681113,
    5.7044858086819185e-3},
   {0.22906653611681113, 0.22906653611681113 ,0.506227344977843697,
    5.7044858086819185e-3},
   {0.22906653611681113, 0.22906653611681113 ,0.035639582788534043,
    5.7044858086819185e-3},
   {0.736298458958971704, 0.19048604193463345 ,0.036607749553197423,
    2.1405191411620925e-3}, // w6
   {0.736298458958971704, 0.036607749553197423 ,0.19048604193463345,
    2.1405191411620925e-3},
   {0.736298458958971704, 0.036607749553197423 ,0.036607749553197423,
    2.1405191411620925e-3},
   {0.19048604193463345, 0.736298458958971704 ,0.036607749553197423,
    2.1405191411620925e-3},
   {0.19048604193463345, 0.036607749553197423 ,0.736298458958971704,
    2.1405191411620925e-3},
   {0.19048604193463345, 0.036607749553197423 ,0.036607749553197423,
    2.1405191411620925e-3},
   {0.036607749553197423, 0.736298458958971704 ,0.19048604193463345,
    2.1405191411620925e-3},
   {0.036607749553197423, 0.736298458958971704 ,0.036607749553197423,
    2.1405191411620925e-3},
   {0.036607749553197423, 0.19048604193463345 ,0.736298458958971704,
    2.1405191411620925e-3},
   {0.036607749553197423, 0.19048604193463345 ,0.036607749553197423,
    2.1405191411620925e-3},
   {0.036607749553197423, 0.036607749553197423 ,0.736298458958971704,
    2.1405191411620925e-3},
   {0.036607749553197423, 0.036607749553197423 ,0.19048604193463345,
    2.1405191411620925e-3}
  };
const Quadrature3d pardg::quad3d_8(43, 8, quad3d_8_x);





// M. Beckers and A. Haegemans,
// The construction of cubature formulae for the tetrahedron,
// Report TW 128, Dept. of Computer Science, K.U. Leuven, 1990.
//
// M. Beckers,  Numerical integration in high dimensions,
// Ph.D. thesis, Katholieke Universiteit Leuven, 1992.
//
// http://www.cs.kuleuven.ac.be/~nines/research/ecf/mtables.html
// login=encyclop, passwd=stroud71
//
// note: has negative baryzentric coordinates!
//       has negative weights
static const double quad3d_9_x[][4] =
  {{0.25, 0.25, 0.25,
    -0.13779903832610864}, // w0
   {0.854946884350789780, 0.048351038549736740 ,0.048351038549736740,
    1.8653365690852895e-3}, // w1
   {0.048351038549736740, 0.854946884350789780 ,0.048351038549736740,
    1.8653365690852895e-3},
   {0.048351038549736740, 0.048351038549736740 ,0.854946884350789780,
    1.8653365690852895e-3},
   {0.048351038549736740, 0.048351038549736740 ,0.048351038549736740,
    1.8653365690852895e-3},
   {0.02626215964635292, 0.32457928011788236 ,0.32457928011788236,
    4.3094239694934006e-3}, // w2
   {0.32457928011788236, 0.02626215964635292 ,0.32457928011788236,
    4.3094239694934006e-3},
   {0.32457928011788236, 0.32457928011788236 ,0.02626215964635292,
    4.3094239694934006e-3},
   {0.32457928011788236, 0.32457928011788236 ,0.32457928011788236,
    4.3094239694934006e-3},
   {0.65615037932801437, 0.11461654022399521 ,0.11461654022399521,
    -0.090184766481201525}, // w3
   {0.11461654022399521, 0.65615037932801437 ,0.11461654022399521,
    -0.090184766481201525},
   {0.11461654022399521, 0.11461654022399521 ,0.65615037932801437,
    -0.090184766481201525},
   {0.11461654022399521, 0.11461654022399521 ,0.11461654022399521,
    -0.090184766481201525},
   {0.32353014426545827, 0.22548995191151391 ,0.22548995191151391,
    0.044672576202511444}, // w4
   {0.22548995191151391, 0.32353014426545827 ,0.22548995191151391,
    0.044672576202511444},
   {0.22548995191151391, 0.22548995191151391 ,0.32353014426545827,
    0.044672576202511444},
   {0.22548995191151391, 0.22548995191151391 ,0.22548995191151391,
    0.044672576202511444},
   {0.653079679889075433, 0.083664701617184967 ,0.13162780924686980,
    0.034700405884550761}, // w5
   {0.653079679889075433, 0.13162780924686980 ,0.083664701617184967,
    0.034700405884550761},
   {0.653079679889075433, 0.13162780924686980 ,0.13162780924686980,
    0.034700405884550761},
   {0.083664701617184967, 0.653079679889075433 ,0.13162780924686980,
    0.034700405884550761},
   {0.083664701617184967, 0.13162780924686980 ,0.653079679889075433,
    0.034700405884550761},
   {0.083664701617184967, 0.13162780924686980 ,0.13162780924686980,
    0.034700405884550761},
   {0.13162780924686980, 0.653079679889075433 ,0.083664701617184967,
    0.034700405884550761},
   {0.13162780924686980, 0.653079679889075433 ,0.13162780924686980,
    0.034700405884550761},
   {0.13162780924686980, 0.083664701617184967 ,0.653079679889075433,
    0.034700405884550761},
   {0.13162780924686980, 0.083664701617184967 ,0.13162780924686980,
    0.034700405884550761},
   {0.13162780924686980, 0.13162780924686980 ,0.653079679889075433,
    0.034700405884550761},
   {0.13162780924686980, 0.13162780924686980 ,0.083664701617184967,
    0.034700405884550761},
   {0.02432721762775785, 0.10776985954942861 ,0.43395146141140677,
    3.3525839026606469e-3}, // w6
   {0.02432721762775785, 0.43395146141140677 ,0.10776985954942861,
    3.3525839026606469e-3},
   {0.02432721762775785, 0.43395146141140677 ,0.43395146141140677,
    3.3525839026606469e-3},
   {0.10776985954942861, 0.02432721762775785 ,0.43395146141140677,
    3.3525839026606469e-3},
   {0.10776985954942861, 0.43395146141140677 ,0.02432721762775785,
    3.3525839026606469e-3},
   {0.10776985954942861, 0.43395146141140677 ,0.43395146141140677,
    3.3525839026606469e-3},
   {0.43395146141140677, 0.02432721762775785 ,0.10776985954942861,
    3.3525839026606469e-3},
   {0.43395146141140677, 0.02432721762775785 ,0.43395146141140677,
    3.3525839026606469e-3},
   {0.43395146141140677, 0.10776985954942861 ,0.02432721762775785,
    3.3525839026606469e-3},
   {0.43395146141140677, 0.10776985954942861 ,0.43395146141140677,
    3.3525839026606469e-3},
   {0.43395146141140677, 0.43395146141140677 ,0.02432721762775785,
    3.3525839026606469e-3},
   {0.43395146141140677, 0.43395146141140677 ,0.10776985954942861,
    3.3525839026606469e-3},
   {0.72619908199946906140, 0.27655347263680734 ,-1.3762773181382007e-3,
    4.3162887555699692e-4},// w7
   {0.72619908199946906140, -1.3762773181382007e-3 ,0.27655347263680734,
    4.3162887555699692e-4},
   {0.72619908199946906140, -1.3762773181382007e-3 ,-1.3762773181382007e-3,
    4.3162887555699692e-4},
   {0.27655347263680734, 0.72619908199946906140 ,-1.3762773181382007e-3,
    4.3162887555699692e-4},
   {0.27655347263680734, -1.3762773181382007e-3 ,0.72619908199946906140,
    4.3162887555699692e-4},
   {0.27655347263680734, -1.3762773181382007e-3 ,-1.3762773181382007e-3,
    4.3162887555699692e-4},
   {-1.3762773181382007e-3, 0.72619908199946906140 ,0.27655347263680734,
    4.3162887555699692e-4},
   {-1.3762773181382007e-3, 0.72619908199946906140 ,-1.3762773181382007e-3,
    4.3162887555699692e-4},
   {-1.3762773181382007e-3, 0.27655347263680734 ,0.72619908199946906140,
    4.3162887555699692e-4},
   {-1.3762773181382007e-3, 0.27655347263680734 ,-1.3762773181382007e-3,
    4.3162887555699692e-4},
   {-1.3762773181382007e-3, -1.3762773181382007e-3 ,0.72619908199946906140,
    4.3162887555699692e-4},
   {-1.3762773181382007e-3, -1.3762773181382007e-3 ,0.27655347263680734,
    4.3162887555699692e-4}
  };
const Quadrature3d pardg::quad3d_9(53, 9, quad3d_9_x);




// Sainz de la Maza, Eduardo;  Maeztu, Jos I.
// An invariant quadrature rule of degree 11 for the tetrahedron.
// (English. Abridged French version)
// C. R. Acad. Sci., Paris, Sr. I 321, No.9, 1263-1267 (1995).
//
// remark: - has negative baryzentric coordinates! and
//           negative weights
//         - coordinates are not computed very accurately
//           e.g. the sum of the baryzentric coordinates of the second group
//           gives
//           3*0.3197881306061907 + 0.04063561442097275
//            = 1.00000000623954485,
//           hence errors can not be better than 1e-9 or 1e-10...
static const double quad3d_11_x[][4] =
  {{0.25, 0.25, 0.25,
    -0.3229059250896649/6.0}, // w0
   {0.3197881306061907, 0.3197881306061907, 0.3197881306061907,
    -0.3831136086645949/6.0}, // w1
   {0.4063561442097275e-1, 0.3197881306061907, 0.3197881306061907,
    -0.3831136086645949/6.0},
   {0.3197881306061907, 0.4063561442097275e-1, 0.3197881306061907,
    -0.3831136086645949/6.0},
   {0.3197881306061907, 0.3197881306061907, 0.4063561442097275e-1,
    -0.3831136086645949/6.0},
   {0.2745875432484354, 0.2745875432484354, 0.2745875432484354,
    0.1259876832639002/6.0}, // w2
   {0.1762373724529911, 0.2745875432484354, 0.2745875432484354,
    0.1259876832639002/6.0},
   {0.2745875432484354, 0.1762373724529911, 0.2745875432484354,
    0.1259876832639002/6.0},
   {0.2745875432484354, 0.2745875432484354, 0.1762373724529911,
    0.1259876832639002/6.0},
   {0.4902463231623282e-1, 0.4902463231623282e-1, 0.4902463231623282e-1,
    0.7772656110490364e-2/6.0}, // w3
   {0.8529260850827039, 0.4902463231623282e-1, 0.4902463231623282e-1,
    0.7772656110490364e-2/6.0},
   {0.4902463231623282e-1, 0.8529260850827039, 0.4902463231623282e-1,
    0.7772656110490364e-2/6.0},
   {0.4902463231623282e-1, 0.4902463231623282e-1, 0.8529260850827039,
    0.7772656110490364e-2/6.0},
   {-0.58892050323316550e-1,-0.58892050323316550e-1,-0.58892050323316550e-1,
    0.4475842042017354e-5/6.0}, // w4
   {0.11766761233528490e1, -0.58892050323316550e-1, -0.58892050323316550e-1,
    0.4475842042017354e-5/6.0},
   {-0.58892050323316550e-1, 0.11766761233528490e1, -0.58892050323316550e-1,
    0.4475842042017354e-5/6.0},
   {-0.58892050323316550e-1, -0.58892050323316550e-1, 0.11766761233528490e1,
    0.4475842042017354e-5/6.0},
   {0.14369806508030763, 0.14369806508030763, 0.14369806508030763,
    0.3076630972851224e-1/6.0}, // w5
   {0.56890579525494376, 0.14369806508030763, 0.14369806508030763,
    0.3076630972851224e-1/6.0},
   {0.14369806508030763, 0.56890579525494376, 0.14369806508030763,
    0.3076630972851224e-1/6.0},
   {0.14369806508030763, 0.14369806508030763, 0.56890579525494376,
    0.3076630972851224e-1/6.0},
   {0.43340593206769717, 0.43340593206769717, 0.66594067932302825e-1,
    0.2230322290225118e-1/6.0}, // w6
   {0.43340593206769717, 0.66594067932302825e-1, 0.43340593206769717,
    0.2230322290225118e-1/6.0},
   {0.66594067932302825e-1, 0.43340593206769717, 0.43340593206769717,
    0.2230322290225118e-1/6.0},
   {0.66594067932302825e-1, 0.43340593206769717, 0.66594067932302825e-1,
    0.2230322290225118e-1/6.0},
   {0.66594067932302825e-1, 0.66594067932302825e-1, 0.43340593206769717,
    0.2230322290225118e-1/6.0},
   {0.43340593206769717, 0.66594067932302825e-1, 0.66594067932302825e-1,
    0.2230322290225118e-1/6.0},
   {0.60987466974805193e-1, -0.67354325781295417e-1 ,0.50318342940324511,
    0.5167456484634155e-3/6.0}, //w7
   {0.60987466974805193e-1, 0.50318342940324511 ,-0.67354325781295417e-1,
    0.5167456484634155e-3/6.0},
   {0.60987466974805193e-1, 0.50318342940324511 ,0.50318342940324511,
    0.5167456484634155e-3/6.0},
   {-0.67354325781295417e-1, 0.60987466974805193e-1 ,0.50318342940324511,
    0.5167456484634155e-3/6.0},
   {-0.67354325781295417e-1, 0.50318342940324511 ,0.60987466974805193e-1,
    0.5167456484634155e-3/6.0},
   {-0.67354325781295417e-1, 0.50318342940324511 ,0.50318342940324511,
    0.5167456484634155e-3/6.0},
   {0.50318342940324511, 0.60987466974805193e-1 ,-0.67354325781295417e-1,
    0.5167456484634155e-3/6.0},
   {0.50318342940324511, 0.60987466974805193e-1 ,0.50318342940324511,
    0.5167456484634155e-3/6.0},
   {0.50318342940324511, -0.67354325781295417e-1 ,0.60987466974805193e-1,
    0.5167456484634155e-3/6.0},
   {0.50318342940324511, -0.67354325781295417e-1 ,0.50318342940324511,
    0.5167456484634155e-3/6.0},
   {0.50318342940324511, 0.50318342940324511 ,0.60987466974805193e-1,
    0.5167456484634155e-3/6.0},
   {0.50318342940324511, 0.50318342940324511 ,-0.67354325781295417e-1,
    0.5167456484634155e-3/6.0},
   {0.37182110608410947, 0.39266554926037518e-1 ,0.29445616949492650,
    0.1484538986489890/6.0}, // w8
   {0.37182110608410947, 0.29445616949492650 ,0.39266554926037518e-1,
    0.1484538986489890/6.0},
   {0.37182110608410947, 0.29445616949492650 ,0.29445616949492650,
    0.1484538986489890/6.0},
   {0.39266554926037518e-1, 0.37182110608410947 ,0.29445616949492650,
    0.1484538986489890/6.0},
   {0.39266554926037518e-1, 0.29445616949492650 ,0.37182110608410947,
    0.1484538986489890/6.0},
   {0.39266554926037518e-1, 0.29445616949492650 ,0.29445616949492650,
    0.1484538986489890/6.0},
   {0.29445616949492650, 0.37182110608410947 ,0.39266554926037518e-1,
    0.1484538986489890/6.0},
   {0.29445616949492650, 0.37182110608410947 ,0.29445616949492650,
    0.1484538986489890/6.0},
   {0.29445616949492650, 0.39266554926037518e-1 ,0.37182110608410947,
    0.1484538986489890/6.0},
   {0.29445616949492650, 0.39266554926037518e-1 ,0.29445616949492650,
    0.1484538986489890/6.0},
   {0.29445616949492650, 0.29445616949492650 ,0.37182110608410947,
    0.1484538986489890/6.0},
   {0.29445616949492650, 0.29445616949492650 ,0.39266554926037518e-1,
    0.1484538986489890/6.0},
   {0.86161014690973263, 0.13838985309026736 ,0.0,
    0.9330967352789100e-3/6.0}, // w9
   {0.86161014690973263, 0.0 ,0.13838985309026736,
    0.9330967352789100e-3/6.0},
   {0.86161014690973263, 0.0 ,0.0,
    0.9330967352789100e-3/6.0},
   {0.13838985309026736, 0.86161014690973263 ,0.0,
    0.9330967352789100e-3/6.0},
   {0.13838985309026736, 0.0 ,0.86161014690973263,
    0.9330967352789100e-3/6.0},
   {0.13838985309026736, 0.0 ,0.0,
    0.9330967352789100e-3/6.0},
   {0.0, 0.86161014690973263 ,0.13838985309026736,
    0.9330967352789100e-3/6.0},
   {0.0, 0.86161014690973263 ,0.0,
    0.9330967352789100e-3/6.0},
   {0.0, 0.13838985309026736 ,0.86161014690973263,
    0.9330967352789100e-3/6.0},
   {0.0, 0.13838985309026736 ,0.0,
    0.9330967352789100e-3/6.0},
   {0.0, 0.0 ,0.86161014690973263,
    0.9330967352789100e-3/6.0},
   {0.0, 0.0 ,0.13838985309026736,
    0.9330967352789100e-3/6.0},
   {0.69267352508351802, 0.16320147746405826e-1 ,0.14550316358503807,
    0.9319130804165715e-2/6.0}, // w10
   {0.69267352508351802, 0.14550316358503807 ,0.16320147746405826e-1,
    0.9319130804165715e-2/6.0},
   {0.69267352508351802, 0.14550316358503807 ,0.14550316358503807,
    0.9319130804165715e-2/6.0},
   {0.16320147746405826e-1, 0.69267352508351802 ,0.14550316358503807,
    0.9319130804165715e-2/6.0},
   {0.16320147746405826e-1, 0.14550316358503807 ,0.69267352508351802,
    0.9319130804165715e-2/6.0},
   {0.16320147746405826e-1, 0.14550316358503807 ,0.14550316358503807,
    0.9319130804165715e-2/6.0},
   {0.14550316358503807, 0.69267352508351802 ,0.16320147746405826e-1,
    0.9319130804165715e-2/6.0},
   {0.14550316358503807, 0.69267352508351802 ,0.14550316358503807,
    0.9319130804165715e-2/6.0},
   {0.14550316358503807, 0.16320147746405826e-1 ,0.69267352508351802,
    0.9319130804165715e-2/6.0},
   {0.14550316358503807, 0.16320147746405826e-1 ,0.14550316358503807,
    0.9319130804165715e-2/6.0},
   {0.14550316358503807, 0.14550316358503807 ,0.69267352508351802,
    0.9319130804165715e-2/6.0},
   {0.14550316358503807, 0.14550316358503807 ,0.16320147746405826e-1,
    0.9319130804165715e-2/6.0},
   {0.63469493926752182, 0.27759599714708815 ,0.43854531792695007e-1,
    0.1272850504266610e-1/6.0}, // w11
   {0.63469493926752182, 0.43854531792695007e-1 ,0.27759599714708815,
    0.1272850504266610e-1/6.0},
   {0.63469493926752182, 0.43854531792695007e-1 ,0.43854531792695007e-1,
    0.1272850504266610e-1/6.0},
   {0.27759599714708815, 0.63469493926752182 ,0.43854531792695007e-1,
    0.1272850504266610e-1/6.0},
   {0.27759599714708815, 0.43854531792695007e-1 ,0.63469493926752182,
    0.1272850504266610e-1/6.0},
   {0.27759599714708815, 0.43854531792695007e-1 ,0.43854531792695007e-1,
    0.1272850504266610e-1/6.0},
   {0.43854531792695007e-1, 0.63469493926752182 ,0.27759599714708815,
    0.1272850504266610e-1/6.0},
   {0.43854531792695007e-1, 0.63469493926752182 ,0.43854531792695007e-1,
    0.1272850504266610e-1/6.0},
   {0.43854531792695007e-1, 0.27759599714708815 ,0.63469493926752182,
    0.1272850504266610e-1/6.0},
   {0.43854531792695007e-1, 0.27759599714708815 ,0.43854531792695007e-1,
    0.1272850504266610e-1/6.0},
   {0.43854531792695007e-1, 0.43854531792695007e-1 ,0.63469493926752182,
    0.1272850504266610e-1/6.0},
   {0.43854531792695007e-1, 0.43854531792695007e-1 ,0.27759599714708815,
    0.1272850504266610e-1/6.0}
  };
const Quadrature3d pardg::quad3d_11(87, 11, quad3d_11_x);

