// This file is part of the dune-xt-functions project:
//   https://github.com/dune-community/dune-xt-functions
// The copyright lies with the authors of this file (see below).
// License: Dual licensed as  BSD 2-Clause License (http://opensource.org/licenses/BSD-2-Clause)
//      or  GPL-2.0+ (http://opensource.org/licenses/gpl-license)
//          with "runtime exception" (http://www.dune-project.org/license.html)
// Authors:
//   Felix Schindler (2013 - 2016)
//   Rene Milk       (2013 - 2015)
//   Sven Kaulmann   (2013)
//   Tobias Leibner  (2014)

#ifndef DUNE_XT_FUNCTIONS_GLOBAL_HH
#define DUNE_XT_FUNCTIONS_GLOBAL_HH

#include <functional>

#include <dune/xt/common/memory.hh>

#include <dune/xt/functions/interfaces.hh>

namespace Dune {
namespace XT {
namespace Functions {

/**
 * Global-valued function you can pass a lambda expression to that gets evaluated
 * \example LambdaType lambda([](DomainType x) { return x;}, 1 );
 */
template <class EntityImp, class DomainFieldImp, size_t domainDim, class RangeFieldImp, size_t rangeDim,
          size_t rangeDimCols = 1>
class GlobalLambdaFunction
    : public GlobalFunctionInterface<EntityImp, DomainFieldImp, domainDim, RangeFieldImp, rangeDim, rangeDimCols>
{
  typedef GlobalFunctionInterface<EntityImp, DomainFieldImp, domainDim, RangeFieldImp, rangeDim, rangeDimCols> BaseType;

public:
  typedef typename BaseType::DomainType DomainType;
  typedef typename BaseType::RangeType RangeType;

private:
  typedef std::function<RangeType(DomainType)> LambdaType;

public:
  GlobalLambdaFunction(LambdaType lambda, const size_t order_in, const std::string nm = "globallambdafunction")
    : lambda_(lambda)
    , order_(order_in)
    , name_(nm)
  {
  }

  virtual size_t order() const override final
  {
    return order_;
  }

  virtual void evaluate(const DomainType& xx, RangeType& ret) const override final
  {
    ret = lambda_(xx);
  }

  virtual RangeType evaluate(const DomainType& xx) const override final
  {
    return lambda_(xx);
  }

  virtual std::string type() const override
  {
    return "globallambdafunction";
  }

  virtual std::string name() const override
  {
    return name_;
  }

private:
  const LambdaType lambda_;
  const size_t order_;
  const std::string name_;
};

} // namespace Functions
} // namespace XT
} // namespace Dune

#endif // DUNE_XT_FUNCTIONS_GLOBAL_HH
